﻿(function () {
	'use strict';
	let DataTable = $.fn.dataTable;

	/* Set the defaults for DataTables initialisation */
	$.extend(true, DataTable.defaults, {
		column: { render: $.fn.dataTable.render.text() },
		orderMulti: false,
		filter: false,
		lengthChange: false,
		language: {
			decimal: "",
			emptyTable: "Нет данных для отображения",
			info: "Всего: _TOTAL_",
			infoEmpty: "Всего: 0",
			infoFiltered: "(filtered from _MAX_ total entries)",
			infoPostFix: "",
			thousands: ",",
			lengthMenu: "_MENU_",
			loadingRecords: "Загрузка...",
			search: "Поиск:",
			zeroRecords: "Совпадающих записей не найдено",
			paginate: {
				first: "First",
				last: "Last",
				sPrevious: " ",
				sNext: " "
			},
			aria: {
				sortAscending: ": активировать для сортировки столбца по возрастанию",
				sortDescending: ": активировать для сортировки столбца по убыванию"
			},
			select: {
				rows: {
					_: "Выбрано: %d",
					0: "Выбрано: 0",
					1: "Выбрано: 1"
				}
			}
		},
		dom:
			"<'data-table-top'f>" +
			"<'data-table-content'tr>" +
			"<'data-table-footer'S<'dt-buttons-container'>lp>",
		renderer: 'redcheck'
	});

	DataTable.defaults.column.orderSequence = ['asc', 'desc'];

	/* Default select settings */
	$(document).on('draw.dt', function (e, settings) {
		const apiSelect = $(settings.nTable).DataTable().select;

		if (apiSelect.style() === 'single') {
			apiSelect.selector('tr');
			apiSelect.toggleable(false);
		}
	});

	$.fn.dataTable.ext.errMode = function (settings, helpPage, message) {
		const lang = getCookie('lang');
		const columnCount = settings.aoColumns.length;
		let text = 'An error occurred during the execution';

		if (lang === 'ru') {
			text = 'В процессе выполнения произошла ошибка';
		}

		console.error(message);

		function showError() {
			$(settings.nTable).find('tbody').html(
				'<tr><td class="dt-empty" colspan="' + columnCount + '">' + text + '</td></tr>'
			);
		}

		showError();

		$(settings.nTable).one('draw.dt', function (e, settings) {
			const api = new $.fn.dataTable.Api(settings);

			if (api.page.info().recordsTotal === 0) {
				showError();
			}
		});
	};

	/* Default class modification */
	$.extend(DataTable.ext.classes, {
		container: "dt-container data-table-container",
		search: {
			container: "dt-search",
			input: "text-input",
		},
		length: {
			container: "dt-length",
			select: "dt-select-box"
		},
		tbody: {
			cell: "",
			row: "data-table__row",
		},
		table: "table dt-loaded",
	});

	DataTable.ext.renderer.pagingButton.redcheck = function (settings, buttonType, content, active, disabled) {
		if (buttonType !== 'next' && buttonType !== 'previous') {
			return {};
		}

		var btnClasses = ['paginate_button'];

		if (active) {
			btnClasses.push('active');
		}

		if (disabled) {
			btnClasses.push('disabled')
		}

		const li = $('<li>').addClass(btnClasses.join(' '));
		const a = $('<a>', {
			'href': disabled ? null : '#',
			'class': 'page-link'
		})
			.text(content)
			.appendTo(li);

		return {
			display: li,
			clicker: a
		};
	};

	DataTable.ext.renderer.pagingContainer.redcheck = function (settings, buttonEls) {
		const api = new DataTable.Api(settings);
		const pageInfo = api.page.info();

		if (pageInfo.pages < 1) {
			return;
		}

		const tableWidth = api.table().body().clientWidth;
		const currentLang = getCookie('lang') || 'en';
		const select = createPaginationSelect(pageInfo.page, pageInfo.pages);
		const info = currentLang === 'ru'
			? (tableWidth < 430 ? 'Стр.' : 'Страница') + ` ${select.outerHTML} из ${pageInfo.pages}`
			: `Page ${select.outerHTML} of ${pageInfo.pages}`;

		const infoElement = $('<span>').html(info);
		const ulElement = $('<ul>').addClass('pagination').append(buttonEls);

		return $('<div/>').append(infoElement).append(ulElement);
	};

	$(document).on('init.dt', function (e, settings) {
		const api = new DataTable.Api(settings);
		const container = api.table().container();

		container.addEventListener('change', function(event) {
			if (event.target.matches('select[id^="dt-page-changer-"]')) {
				const pageNumber = event.target.value;
				api.page(pageNumber - 1).draw('page');
			}
		});
	});

	let pageChangerCounter = 0;
	function createPaginationSelect(page, pages) {
		const select = document.createElement('select');
		select.classList.add('dt-select-box');
		select.id = 'dt-page-changer-' + pageChangerCounter;
		pageChangerCounter++;

		const options = Array.from({ length: pages }, (v, i) => i + 1);
		options.forEach(value => {
			const option = document.createElement('option');
			option.value = value;
			option.textContent = value;

			if (value === page + 1) {
				option.setAttribute('selected', 'true');
			}

			select.appendChild(option);
		});
		
		return select;
	}

	// Render helper to display a list of hosts
	$.fn.dataTable.render.hostList = function () {
		return function (targets, type, full, meta) {
			const lang = getCookie('lang') || 'en';

			if (full.CountOfHosts === 0) return '';

			if (full.CountOfHosts > 5 || targets == null) {
				let span = document.createElement('span');
				span.classList.add('rc_label');
				span.classList.add('padding-0');
				span.textContent = '( ' + full.CountOfHosts + ' ' + (lang === 'en' ? 'items' : 'элементов') + ' )';

				return span.outerHTML;
			}

			let div = document.createElement('div');
			div.classList.add('data-table-cell_host-list');

			targets.forEach(function (host) {
				let span = document.createElement('span');
				span.classList.add('rc_label');
				span.classList.add('gray');
				span.dataset.id = host.Name;
				span.textContent = host.Name;

				div.appendChild(span);
			});

			return div.outerHTML;
		};
	};

    /* Rendering helper for displaying the date and time */
	$.fn.dataTable.render.date = function () {
		return function (data, type, row, meta) {
			const lang = getCookie('lang') || 'en';

			if (data) {
				return new Date(data).toLocaleString(lang);
            }

			return '';
		};
	};

	/* Rendering helper for displaying the date only */
	$.fn.dataTable.render.dateOnly = function () {
		return function (data, type, row, meta) {
			const lang = getCookie('lang') || 'en';

			if (data) {
				return new Date(data).toLocaleDateString(lang);
			}

			return '';
		};
	};


	/* Click event handler for command buttons */
	$(document).on('draw.dt', function (e, settings) {
		let list = settings.nTable.querySelectorAll('a[id*=cmd-link-id-]');
		for (var i = 0; i < list.length; i++) {
			let hash = list[i].id.split('cmd-link-id-')[1];
			list[i].addEventListener('click', function (event) {
				event.preventDefault();
				openCommands(hash);
			});
		}

		$(settings.nTable).find('.command').on('click', function (event) {
			event.stopPropagation();
			$('body').trigger('click');

			let sender = event.currentTarget;
			let handler = $(sender).data('handler');

			if (handler) {
				event.preventDefault();

				let index = $(sender).data('row-index');
				let api = $(sender).closest('table').DataTable();

				// Сalling and checking the existence of a function on the client
				handler.split('.').reduce(function (acc, functionName) {
					if (typeof acc[functionName] === 'function' || typeof acc[functionName] === 'object') {
						return acc[functionName];
					} else {
						throw 'Failed to get event handler for ' + handler + '. ' + functionName + ' not found';
					}
				}, window)(sender, api.row(index).data());
			}
		});
	});

	/* Rendering helper for displaying the command buttons */
	$.fn.dataTable.render.commandButtons = function () {
		function createItem(params, meta) {
			let attr = '';

			if (params.hasOwnProperty("Url") && params.Url) {
				attr += ' href="' + params.Url + '"';
			} else if (params.OnClick) {
				attr += ' href="#"';
			}

			if (params.OnClick) {
				attr += ' data-handler="' + params.OnClick + '"';
				attr += ' data-table-id="' + meta.settings.sTableId + '"';
				attr += ' data-row-index="' + meta.row + '"';
			}

			let cmd = [
				'<a class="command" ' + attr + ' name="command-link">',
				params.IconPath ? '<img class="command-icons" src="' + params.IconPath + '">' : '',
				'<span class="command-text">' + params.Title + '</span>',
				'</a>'
			].join('');

			if (!params.IsActive) {
				cmd = '<div class="disableContent">' + cmd + '</div>'
			}

			return cmd;
		}

		function createSubMenu(data, meta) {
			let commands = data.reduce(function (acc, params) {
				return acc + createItem(params, meta);
			}, '');

			return '<div class="popover-sub-menu-content">' + commands + '</div>';
		}

		return function (data, type, row, meta) {
			let hash = hashString(JSON.stringify(data));

			let commands = data.reduce(function (acc, params) {
				let result = '';

				if (!params.Children) {
					result = createItem(params, meta);
				} else {
					const item = createItem(params, meta);
					const subMenu = createSubMenu(params.Children, meta);

					result = '<div class="popover-sub-menu">' + item + subMenu + '</div>';
				}

				return acc + result;
			}, '');

			let html = [
				'<a class="command-header" id="cmd-link-id-' + hash + '" name="command-header" href="#">',
					'<img class="icons" name="cmd-icons" height="18" src="/img/rc_icons/settings.png">',
				'</a>',
				'<div class="popover fade left in" id="command-id-' + hash + '">',
					commands,
				'</div>'
			].join('');

			return html;
		};
	};

	/* Align column widths for all tables */
	$.fn.dataTable.util.columnsAjustSize = function (inludeHidden) {
		inludeHidden = inludeHidden === undefined ? true : inludeHidden;

		$.fn.dataTable.tables().forEach(function (table) {
			if (inludeHidden === true) {
				$(table).DataTable().columns.adjust();
			} else if ($(table).is(':visible')) {
				$(table).DataTable().columns.adjust();
			}
		});
	};

	// Registering the 'newSearch' event
	$(document).on('init.dt', function (e, settings) {
		if (settings.oInit.searching) {
			const api = new DataTable.Api(settings);

			settings._snapshotSearch = getSearchValueOfColumns(settings);
			api.on('preDraw.dt', onSearchObserver);
			api.on('destroy.dt', function () {
				api.off('preDraw.dt', onSearchObserver);
			});
        }
	});

	function onSearchObserver(e, settings) {
		const searchValues = getSearchValueOfColumns(settings);

		const isEqual = searchValues.every(function (text, i) {
			return settings._snapshotSearch[i] === text;
		});

		settings._snapshotSearch = searchValues;

		if (!isEqual) {
			$(settings.nTable).trigger('newSearch.dt', [settings]);
		}
	}

	function getSearchValueOfColumns(settings) {
		return settings.aoPreSearchCols.map(function (column) {
			return column.sSearch;
		});
	}

	$(document).on('preInit.dt', function (e, settings) {
		const api = new $.fn.dataTable.Api(settings);
		$(api.table().header()).find('[data-class]').each(function() {
			const dataClassValue = $(this).attr('data-class');
			$(this).addClass(dataClassValue);
		});
	});
}());

(function () {
	// Selection module

	var DataTable = $.fn.dataTable;

	function init(dt) {
		const ctx = dt.settings()[0];
		const elmSelectAll = ctx.nTHead.querySelector('th.select-checkbox');
		const defaults = {
			primaryKey: 'Id:name',
			checkBoxesMode: 'none',
			selectAllMode: 'none', // all, page, none
			resetSelectionAfterSearch: true,
			onSelectionChanged: function (e, ids) {}
		};

		const initParam = ctx.oInit.selectCustom;
		let opts = initParam === undefined ? {} : initParam;

		opts.primaryKey = opts.primaryKey || defaults.primaryKey;
		opts.checkBoxesMode = opts.checkBoxesMode || defaults.checkBoxesMode;
		opts.selectAllMode = elmSelectAll ? 'page' : defaults.selectAllMode;
		opts.resetSelectionAfterSearch = opts.resetSelectionAfterSearch !== undefined
			? opts.resetSelectionAfterSearch : defaults.resetSelectionAfterSearch;
		opts.onSelectionChanged = opts.onSelectionChanged || defaults.onSelectionChanged;
		opts.allItemKeys = [];
		opts.selectedItemKeys = [];

		ctx._selectCustom = opts;

		dt.on('newSearch.dt', onNewSearch);
		dt.on('draw.dt', onDrawTable);
		dt.on('deselect.dt', onDeselectRows);
		dt.on('select.dt', onSelectRows);

		// Executed after all changes
		dt.on('select.dt', onSelectionChanged);

		if (elmSelectAll)
			elmSelectAll.addEventListener('click', handlerSelectAll.bind(dt));
	}

	function onSelectRows(e, api, type, indexes) {
		const style = api.settings()[0]._select.style;
		const rowData = api.rows(indexes).data().toArray();
		let selectData = api.settings()[0]._selectCustom;

		if (style === 'single' && rowData.length) {
			selectData.selectedItemKeys = [rowData[0].Id];
			return;
		}

		for (let i = 0; i < rowData.length; i++) {
			const row = rowData[i];

			if (selectData.selectedItemKeys.indexOf(row.Id) > -1) {
				continue;
			}
			selectData.selectedItemKeys.push(row.Id);
		}
	}

	function onDrawTable(e, settings) {
		const api = new DataTable.Api(settings);
		const columnSelector = settings._selectCustom.primaryKey;
		const itemKeys = settings._selectCustom.selectedItemKeys;

        const indexes = api.rows().eq(0).filter(function (rowIndex) {
            const key = api.cell(rowIndex, columnSelector).data();

			for (let i = 0; i < itemKeys.length; i++) {
				if (itemKeys[i] === key) return true;
			}
			return false;
		});

		api.rows(indexes).select();
		updateInfo(settings);
	}

	function onNewSearch(e, settings) {
		let selectData = settings._selectCustom;

		if (selectData.resetSelectionAfterSearch === true) {
			selectData.selectedItemKeys = [];
			selectData.allItemKeys = [];
		}
    }

	function onDeselectRows(e, api, type, indexes) {
		const rowData = api.rows(indexes).data().toArray();
		let selectData = api.settings()[0]._selectCustom;

		for (let i = 0; i < rowData.length; i++) {
			const index = selectData.selectedItemKeys.indexOf(rowData[i].Id);
			if (index > -1) {
				selectData.selectedItemKeys.splice(index, 1);
			}
		}

		updateInfo(api.settings()[0]);
	}

	function onSelectionChanged (e, api) {
		const settings = api.settings()[0];
		const selectData = settings._selectCustom;

		updateInfo(settings);
		selectData.onSelectionChanged(null, selectData.selectedItemKeys);
	}

	function updateInfo(settings) {
		const style = settings._select.style;
		const count = settings._selectCustom.selectedItemKeys.length;

		if (style === 'single') {
			return;
        }

		renderSelectAllButton(settings);
	}

	function renderSelectAllButton(settings) {
		const api = new DataTable.Api(settings);
		const selectAllMode = api.selectCustom().selectAllMode();
		const elmSelectAll = settings.nTHead.querySelector('th.select-checkbox');

		if (!elmSelectAll) return;

		const countRows = selectAllMode === 'all' ? api.page.info().recordsTotal : api.page.info().length;
		const selectedRows = selectAllMode === 'all' ? api.selectCustom().selectedItemKeys().length : api.rows({selected: true}).count();

		elmSelectAll.classList.remove('select-checkbox__active');
		elmSelectAll.classList.remove('select-checkbox__secondary');

		if (countRows === 0 || selectedRows === 0) return;
		
		if (countRows !== selectedRows) {
			elmSelectAll.classList.add('select-checkbox__secondary');
		} else {
			elmSelectAll.classList.add('select-checkbox__active');
		}
	}

	function handlerSelectAll(e) {
		const api = new DataTable.Api(this);
		const selectAllMode = api.selectCustom().selectAllMode();
		
		const countRows = selectAllMode === 'all' ? api.page.info().recordsTotal : api.page.info().length;
		const selectedRows = selectAllMode === 'all' ? api.selectCustom().selectedItemKeys().length : api.rows({selected: true}).count();

		if (countRows === 0) return;
		
		if (selectedRows === 0) {
			if (selectAllMode === 'all') {
				api.processing(true);
				api.selectCustom().selectAll().done(() => api.draw(false));
			} else if (selectAllMode === 'page') {
				api.rows().select();
			}
		} else {
			if (selectAllMode === 'all') {
				api.selectCustom().deselectAll();
				api.draw(false);
			} else if (selectAllMode === 'page') {
				api.rows().deselect();
			}
		}
	}

	function getAllItemKeys(url, data, callback) {
		return $.ajax({
			type: 'POST',
			contentType: 'application/x-www-form-urlencoded; charset=utf-8',
			url: url,
			data: data,
			dataType: 'json'
		}).done(callback);
    }

	function apiRegister() {
		DataTable.Api.register('selectCustom()', function () {
			return this;
		});

		DataTable.Api.register('selectCustom().selectAllMode()', function () {
			return this.settings()[0]._selectCustom.selectAllMode;
		});

		DataTable.Api.register('selectCustom().selectedItemKeys()', function () {
			const selectData = this.settings()[0]._selectCustom;
			return selectData ? selectData.selectedItemKeys : [];
		});

		DataTable.Api.register('selectCustom().selectByKeys()', function (keys) {
			const selectData = this.settings()[0]._selectCustom;
			selectData.selectedItemKeys = keys;
			return this;
		});

		DataTable.Api.register('selectCustom().selectAll()', function () {
			const api = this;
			const selectData = api.settings()[0]._selectCustom;
			const deferred = new $.Deferred();

			if (!selectData.allItemKeys.length && selectData.urlItemsId) {
				getAllItemKeys(selectData.urlItemsId, api.ajax.params(), function (data) {
					selectData.allItemKeys = data.data;
					selectData.selectedItemKeys = selectData.allItemKeys.concat([]);
					deferred.resolve();
				});
			} else {
				selectData.selectedItemKeys = selectData.allItemKeys.concat([]);
				deferred.resolve();
            }

			return deferred;
		});

		DataTable.Api.register('selectCustom().deselectAll()', function () {
			const selectData = this.settings()[0]._selectCustom;
			selectData.selectedItemKeys = [];
			return this;
		});
    }

	// Entry point for initializing the extension
	$(document).on('preInit.dt', function (e, settings) {
		if (settings.oInit.selectCustom) {
			init(new DataTable.Api(settings));
		}
		const inputSelectAll = document.querySelector('.dt-checkboxes-select-all input');
		const inputsInTable = document.querySelector('.dt-scroll-headInner table.data-table thead .select-checkbox');
		if (inputSelectAll)
			inputSelectAll.classList.add('rc-input');
		if (inputsInTable)
			inputsInTable.style = 'max-width: 24px;';
	});

	apiRegister();
}());

DataTable.select.classes = {
	checkbox: 'rc-input'
};

DataTable.defaults.useTableIdForRows = false;
DataTable.defaults.rowIdField = 'Id';

$(document).on('options.dt', function (e, options) {
	if (options.useTableIdForRows === true && !options.rowId) {
		const tableId = e.target.id;
		const fieldName = options.rowIdField || 'Id';

		options.rowId = function(rowData) {
			return `${tableId}-row-${rowData[fieldName]}`;
		};
	} else if (options.useTableIdForRows === true && options.rowId) {
		console.warn('DataTables: Both rowId and useTableIdForRows are set. useTableIdForRows will be ignored.');
	}
});

DataTable.Api.register('redcheck.selectedRowCount()', function () {
	const api = this.context[0].api;
	const settings = api.settings()[0];
	
	if (api.selectCustom !== undefined && settings._selectCustom !== undefined && api.select.style() !== 'single') {
		return api.selectCustom().selectedItemKeys().length;
	}

	if (api.column(0).checkboxes !== undefined) {
		const count = api.column(0).checkboxes.selected().length;
		if (count > 0) {
			return count;
		}
	}

	return api.select.cumulative().rows.length;
});

DataTable.Api.register('redcheck.selectedIds()', function () {
	const api = this.context[0].api;

	const tableId = api.table().node().id;
	const prefix = `${tableId}-row-`;

	const rows = api.select.cumulative().rows;
	return rows.map(item => parseInt(item.slice(prefix.length), 10));
});

DataTable.Api.register('redcheck.deselectAll()', function () {
	const api = this.context[0].api;
	const settings = api.settings()[0];
	
	if (api.selectCustom !== undefined && settings._selectCustom !== undefined) {
		api.selectCustom().deselectAll();
		return this;
	}

	if (api.column(0).checkboxes !== undefined) {
		api.column(0).checkboxes.deselectAll();
	}

	api.settings()[0]._select_set = [];
	api.rows().deselect();
	return this;
});

DataTable.Api.register('redcheck.focusedId()', function () {
	const api = this.context[0].api;

	const tableId = api.table().node().id;
	const prefix = `${tableId}-row-`;

	const rowId = api.rowFocus().focusedRowId();
	return rowId !== null ? parseInt(rowId.replace(prefix, ''), 10) : null;
});

function adjustGridSizeInSplitter(tableId) {
	const table = document.getElementById(tableId);

	if (!$.fn.DataTable.isDataTable(table)) return;

    const settings = $(table).dataTable().api().settings()[0];
    const $wraper = $(settings.nTableWrapper);

    const wraperHeight = $wraper.height();
    const parentHeight = $wraper.parent().height();

    if (wraperHeight !== parentHeight) {
        const $scrollBody = $(settings.nScrollBody);
        const height = $scrollBody.height() - (wraperHeight - parentHeight);

        $scrollBody.css('height', height);
        $scrollBody.css('max-height', height);
    }
}

var DataTablesCustomElement = {
	createDetailButton: function (isExpanded) {
		isExpanded = isExpanded || false;
		const button = document.createElement('button');
		button.type = 'button';
		button.classList.add('detail-button-dt');
		button.classList.add('icon');
		button.classList.add(isExpanded ? 'icon_collapse' : 'icon_expand');
		button.setAttribute('aria-label', isExpanded ? 'Collapse' : 'Expand');
		return button;
	},

	toggleDetailButton: function (button, isExpanded) {
		isExpanded = isExpanded || false;
		button.classList.remove('icon_collapse');
		button.classList.remove('icon_expand');

		button.classList.add(isExpanded ? 'icon_collapse' : 'icon_expand');
		button.setAttribute('aria-label', isExpanded ? 'Collapse' : 'Expand');
	},

	createSeverityBadge: function (severityLevel, text, title) {
		severityLevel = parseInt(severityLevel);
		let severityTitle = 'NotAvailable';
		switch (severityLevel) {
			case 1: severityTitle = 'NotAvailable'; break;
			case 2: severityTitle = 'Information'; break;
			case 3: severityTitle = 'Low'; break;
			case 4: severityTitle = 'Medium'; break;
			case 5: severityTitle = 'High'; break;
			case 6: severityTitle = 'Critical'; break;
		}

		const span = document.createElement('span');
		span.classList.add('label_severity');
		span.classList.add(severityTitle);
		span.dataset.id = severityTitle;
		span.dataset.toggle = 'tooltip';
		span.dataset.placement = 'bottom';
		span.dataset.originalTitle = title;
		span.textContent = text;

		return span;
    },

	createComplianceStateBadge: function (complianceState, text, title) {
		complianceStateInt = parseInt(complianceState);
		let complianceStateTitle = 'Unsatisfactory';
		switch (complianceStateInt) {
			case 1: complianceStateTitle = 'Excellent'; break;
			case 2: complianceStateTitle = 'Good'; break;
			case 3: complianceStateTitle = 'Unsatisfactory'; break;
		}

		const span = document.createElement('span');
		span.classList.add('label_compliance_state');
		span.classList.add(complianceStateTitle);
		span.dataset.id = complianceStateTitle;
		span.dataset.toggle = 'tooltip';
		span.dataset.placement = 'bottom';
		span.dataset.originalTitle = title;
		span.textContent = text;

		return span;
	},

	createXResultEnumerationBadge: function (xResultEnumeration, text, title) {
		xResultEnumeration = parseInt(xResultEnumeration);
		let xResultEnumerationTitle = 'Unknown';
		switch (xResultEnumeration) {
			case 0: xResultEnumerationTitle = 'Pass'; break;
			case 1: xResultEnumerationTitle = 'Fail'; break;
			case 2: xResultEnumerationTitle = 'Error'; break;
			case 3: xResultEnumerationTitle = 'Unknown'; break;
			case 4: xResultEnumerationTitle = 'NotApplicable'; break;
			case 5: xResultEnumerationTitle = 'NotChecked'; break;
			case 6: xResultEnumerationTitle = 'NotSelected'; break;
			case 7: xResultEnumerationTitle = 'Informational'; break;
			case 8: xResultEnumerationTitle = 'Fixed'; break;
		}

		const span = document.createElement('span');
		span.classList.add('label_xresultenumeration');
		span.classList.add(xResultEnumerationTitle);
		span.dataset.id = xResultEnumerationTitle;
		span.dataset.toggle = 'tooltip';
		span.dataset.placement = 'bottom';
		span.dataset.originalTitle = title;
		span.textContent = text;

		return span;
	},

	createVulnerabilityStatusBadge: function (vulnerabilityStatus, text, title) {
		vulnerabilityStatus = parseInt(vulnerabilityStatus);
		let vulnerabilityStatusTitle = 'Unknown';
		switch (vulnerabilityStatus) {
			case 1: vulnerabilityStatus = 'New'; break;
			case 2: vulnerabilityStatus = 'Remaining'; break;
			case 3: vulnerabilityStatus = 'Remediated'; break;
			case 4: vulnerabilityStatus = 'Unknown'; break;
		}

		const span = document.createElement('span');
		span.classList.add('label_vulnerability_status');
		span.classList.add(vulnerabilityStatus);
		span.dataset.id = vulnerabilityStatusTitle;
		span.dataset.toggle = 'tooltip';
		span.dataset.placement = 'bottom';
		span.dataset.originalTitle = title;
		span.textContent = text;

		return span;
	},
};

// Plugins

$.fn.dataTable.Api.register('column().title()', function () {
	var colheader = this.header();
	return $(colheader).text().trim();
});

// Features

class TableInfo {
	constructor(settings, options) {
		this.api = new DataTable.Api(settings);
		this.tableId = this.api.table().node().id;

		this.api.on('select.dt deselect.dt draw.dt', () =>
			setTimeout(() => this._render(), 10)
		);

		const footer = this.api.table().container().querySelector('.data-table-footer');
		footer.addEventListener('click', this.onCommandClick.bind(this));
	}

	get canSelectAll() {
		const settings = this.api.settings()[0];
		return settings._selectCustom !== undefined;
	}
	
	get recordsTotal() {
		if (this.api.remoteGrouping().enabled()) {
			return this.api.remoteGrouping().info().recordsTotal;
		}

		return this.api.page.info().recordsTotal;
	}

	onCommandClick(event) {
		const commandLink = event.target.closest('a');
		if (commandLink === null || commandLink.getAttribute('data-command-type') === null) {
			return;
		}

		event.preventDefault();
		this._executeCommand(commandLink.dataset.commandType);
	}

	container() {
		let containerMenu = document.getElementById(`${this.tableId}_footer_menu`);

		if (containerMenu === null) {
			containerMenu = document.createElement("div");
			containerMenu.id = `${this.tableId}_footer_menu`;
			containerMenu.classList.add('dt-info-block');
		}

		return containerMenu;
	}

	_render() {
		const selectedRowCount = this.api.redcheck.selectedRowCount();
		const rowCount = this.recordsTotal;

		const containerMenu = this.container();
		containerMenu.innerHTML = '';

		if (selectedRowCount > 0) {
			this._createMenu(containerMenu, selectedRowCount, rowCount);
		} else {
			const info = document.createElement("span");
			info.textContent = 'Всего: ' + rowCount;
			containerMenu.prepend(info);
		}
	}

	_createMenu(containerMenu, selectedRowCount, rowCount) {
		const hash = hashString(JSON.stringify(containerMenu.id));

		const button = document.createElement("a");
		button.classList.add('dt-info-block__button');
		button.id = 'cmd-link-id-' + hash;
		button.href = '#';
		button.textContent = 'Выбрано: ' + selectedRowCount + ' из ' + rowCount;

		const iconArrow = document.createElement('i');
		iconArrow.className = 'rc-icon rc-icon--chevron-down';
		button.append(iconArrow);

		containerMenu.appendChild(button);

		const menu = `
			<div class="popover fade left in" id="command-id-${hash}">
				<a class="command" href="#" data-command-type="selectAllOnPage">
					<span class="command-text">Выбрать всё на странице</span>
				</a>
				<a class="command ${this.canSelectAll ? '' : 'hide'}" href="#" data-command-type="selectAll">
					<span class="command-text">Выбрать всё (${rowCount})</span>
				</a>
				<div class="command-divider"></div>
				<a class="command" href="#" data-command-type="deselectAll">
					<span class="command-text">Сбросить</span>
				</a>
			</div>
		`;

		containerMenu.innerHTML += menu;
	}

	_executeCommand(type) {
		switch (type) {
			case 'selectAllOnPage':
				this.api.rows().select();
				break;
			case 'selectAll':
				this.api.processing(true);
				this.api.selectCustom().selectAll().done(() => this.api.draw(false));
				break;
			case 'deselectAll':
				this.api.redcheck.deselectAll().draw(false);
				break;
		}
	}
}

DataTable.ext.feature.push({
	fnInit: function (settings, options) {
		return new TableInfo(settings, options).container();
	},
	cFeature: 'S'
});

$(document).on('init.dt', function (e, settings) {
	const api = new DataTable.Api(settings);
	const footer = api.table().container().querySelector('.data-table-footer');

	footer.addEventListener('click', function(event) {
		const target = event.target.closest('a[id*=cmd-link-id-]');

		if (target) {
			event.preventDefault();

			const hash = target.id.split('cmd-link-id-')[1];
			openCommands(hash, 'top left');
		}
	});
});
