﻿var MasterDetail = (function () {
    let _detailButton = DataTablesCustomElement.createDetailButton();

    function MasterDetail(settings, options) {
        const _api = new $.fn.dataTable.Api(settings);

        this.api = _api;
        this.drawTemplate = options.template || function () { return ''; };
        const mode = options.mode || 'auto';

        this.isDetailButton = false;
        this.isClickRow = false;
        this.isDoubleClickRow = false;

        switch (mode) {
            case 'button':
                if (_api.column('DetailButtonColumn:name').any()) {
                    this.isDetailButton = true;
                } else {
                    this.isClickRow = true;
                    console.log('There is no column "DetailButtonColumn". Simplified mode enabled for MasterDetail');
                }
                break;
            default:
                if (_api.column('DetailButtonColumn:name').any()) {
                    this.isDetailButton = true;
                    this.isDoubleClickRow = true;
                } else {
                    this.isClickRow = true;
                }
                break;
        }

        if (this.isDetailButton) {
            _api.on('click', '.detail-button-cell', { context: this }, onClick);
        }

        if (this.isClickRow) {
            _api.on('click', 'td', { context: this }, onClick);
        }

        if (this.isDoubleClickRow) {
            _api.on('dblclick', 'td', { context: this }, onDoubleClick);
        }

        _api.on('draw', { context: this }, onDrawComplete);

        // add listeners RowGroup and CollapsibleGroup
        _api.on('rowgroup-datasrc', onRowGroupDataSrc);
        _api.on('collapsible-group-close', onCollapsibleGroupClose);
    }

    function onDrawComplete(e, settings) {
        const context = e.data.context;
        const api = new $.fn.dataTable.Api(settings);

        if (context.isDetailButton) {
            const tds = api.column('DetailButtonColumn:name').nodes().toArray();

            tds.forEach(function (cell) {
                if (cell.classList.contains('detail-button-cell')) {
                    return;
                }
                cell.classList.add('detail-button-cell');
                cell.innerHTML = _detailButton.outerHTML;
            });
        }
    }

    function onClick(e) {
        const context = e.data.context;

        const row = context.api.row(e.currentTarget);
        if (row.any()) {
            _handlerExpand(row, context);
        }
    }

    function onDoubleClick(e) {
        const context = e.data.context;

        if (window.getSelection) {
            window.getSelection().removeAllRanges();
        }

        if (isColumnToControl(context.api, e.currentTarget)) {
            return;
        }

        const row = context.api.row(e.currentTarget);
        if (row.any()) {
            _handlerExpand(row, context);
        }
    }

    function onRowGroupDataSrc(e, api, value) {
        _hideDetails(null, api);
    }

    function onCollapsibleGroupClose(e, rows, group) {
        _hideDetails(rows, api);
    }

    function _handlerExpand(row, context) {
        const api = context.api;

        if (row.child.isShown()) {
            _hideDetails(null, api);
            return;
        }

        const deferred = new $.Deferred();

        deferred.then(function () {
            api.processing(true);

            return context.drawTemplate(row.data());
        }).always(function (content) {
            _hideDetails(null, api);
            _showDetails(row, content, api);

            api.processing(false);
        });

        deferred.resolve();
    }

    function _hideDetails(rows, api) {
        rows = rows || api.rows();

        rows.every(function () {
            this.node().classList.remove('expanded');
            if (this.child.isShown()) this.child.hide();

            _refreshButton(this.node(), false);
        });
    }

    function _showDetails(row, content, api) {
        row.node().classList.add('expanded');

        const tdContent = document.createElement('td');
        tdContent.setAttribute('colspan', '12');
        tdContent.appendChild(content);

        const rowChild = document.createElement('tr');
        rowChild.classList.add('data-table__row');
        rowChild.classList.add('child');
        rowChild.appendChild(tdContent);

        row.child(rowChild).show();

        const isGrouping = api.remoteGrouping !== undefined ? api.remoteGrouping().enabled() : false
            || api.rowGroup !== undefined ? api.rowGroup().enabled() : false;
        if (isGrouping) {
            $(rowChild).prepend($('<td/>').addClass('group-cell'));
        }

        _refreshButton(row.node(), true);
    }

    function _refreshButton(rowNode, isExpanded) {
        const button = $(rowNode).find('td.detail-button-cell > button').get(0);
        if (button) {
            DataTablesCustomElement.toggleDetailButton(button, isExpanded);
        }
    }

    function isColumnToControl(api, node) {
        const currentColumn = api.column(node);
        const detailButtonColumn = api.column('DetailButtonColumn:name');

        return currentColumn.index() == detailButtonColumn.index();
    }

    return MasterDetail;
}());

(function () {
    $.fn.dataTable.Api.register('masterDetail()', function () {
        return this;
    });

    $.fn.dataTable.Api.register('masterDetail().getColumnIndex()', function () {
        return this.column('DetailButtonColumn:name').index();
    });

    function _init(settings) {
        const options = settings.oInit.masterDetail;
        settings._masterDetail = new MasterDetail(settings, options);
    }

    // Entry point for initializing the extension
    $(document).on('preInit.dt', function (e, settings, json) {
        if (e.namespace !== 'dt') {
            return;
        }

        if (settings.oInit.masterDetail) {
            if (!settings._masterDetail) {
                _init(settings);
            }
        }
    });
}());
