﻿var RemoteGrouping = (function () {
    function RemoteGrouping(settings, options) {
        this.api = new $.fn.dataTable.Api(settings);
        this.enable = options.enable || false;
        this.dataSrc = options.dataSrc || '';
        this.order = options.order;

        this.groups = null;

        if (this.enable === true && this.dataSrc.length === 0) {
            this.enable = false;
        }

        if (options.startRender) {
            this.template = getDefaultTemplate(options.startRender);
        } else {
            this.template = getDefaultTemplate(function (table, group) {
                return group.name + ' (' + group.count + ')';
            });
        }

        this.api.on('draw', { context: this }, onDrawComplete);
        this.api.on('preXhr.dt', { context: this }, onPreXhr);
        $(this.api.table().body()).on('click', 'tr.grouping-row', { context: this }, onClickControl);
    }

    function onDrawComplete(e, settings) {
        const context = e.data.context;
        const api = new $.fn.dataTable.Api(settings);

        const isGrouping = api.remoteGrouping().enabled();
        api.column('GroupSpaceColumn:name').visible(isGrouping);

        if (!isGrouping) {
            return;
        }

        const container = api.table().body();
        const rows = api.rows({ page: 'current' }).nodes();
        let last = null;

        const pageInfo = api.page.info();
        let groupsForPage = getGroupsForPage(context, pageInfo.page);
        const funcRender = getRenderByRow(api, groupsForPage);

        // Hide notification without data
        if (!rows.any() && groupsForPage.length > 0) {
            $(container).html('');
        }

        const columnSelector = context.dataSrc + ':name';
        api.column(columnSelector, { page: 'current' }).data().each(function (groupName, i) {
            if (last !== groupName) {
                const row = $(rows).eq(i);
                funcRender(row, groupName, context.template);
                last = groupName;
            }
        });

        // Displaying other groups
        groupsForPage.forEach(function (group) {
            $(container).append(context.template(api, group));
        });
    }

    function onPreXhr(e, settings, data) {
        const api = new $.fn.dataTable.Api(settings);
        const context = e.data.context;

        const isGrouping = api.remoteGrouping().enabled();

        if (!isGrouping) {
            return;
        }

        const allGroups = context.groups || [];
        const groupKeys = allGroups
            .filter(function (group) { return group.isExpanded; })
            .map(function (group) { return group.name });

        data.grouping = {
            groupBy: api.remoteGrouping().dataSrc(),
            groupKeys: groupKeys
        };
    }

    function onClickControl(e) {
        if (e.target.tagName !== 'BUTTON') return;

        const context = e.data.context;
        const groupId = $(this).data('group-id');
        const group = getGroupById(context, groupId);

        group.isExpanded = !group.isExpanded;
        context.api.draw(false);
    }

    function getRenderByRow(api, groups) {
        let currentGroup = null;

        return function (row, groupName, callbalck) {
            currentGroup = groups.shift();

            while (currentGroup && currentGroup.name != groupName) {
                $(row).before(callbalck(api, currentGroup));
                currentGroup = groups.shift();
            }

            if (currentGroup && currentGroup.name == groupName) {
                $(row).before(callbalck(api, currentGroup));
            } else {
                console.log('Error while splitting rows into categories');
            }
        };
    }

    function getGroupsForPage(context, page) {
        const lastPage = context.api.page.info().pages - 1;
        const pageLength = context.api.page.len();

        const groupList = context.groups;
        let result = [];
        let pos = 0;

        for (let i = 0; i < groupList.length; i++) {
            let group = $.extend({}, groupList[i]);

            group.startPage = Math.floor(pos / pageLength);
            group.endPage = group.isExpanded == true ? Math.floor((pos + group.count - 1) / pageLength) : group.startPage;

            if (page >= group.startPage && page <= group.endPage) {
                result.push(group);
            } else {
                // The group belongs to the next page, but when paginated, it is the last page 
                if (group.startPage > lastPage && page === lastPage) {
                    result.push(group);
                }
            }

            if (group.isExpanded == true) {
                pos += group.count;
            }
        }

        return result;
    }

    function getGroupById(context, groupId) {
        if (groupId < 0 || groupId > context.groups.length - 1) {
            return null;
        }

        return context.groups[groupId];
    }

    function getDefaultTemplate(callback) {
        return function (api, group) {
            const currentPage = api.page.info().page;
            var isExpanded = group.isExpanded;

            const button = DataTablesCustomElement.createDetailButton(isExpanded);
            button.style.display = currentPage > group.startPage ? 'none' : 'inline';
            const content = callback(api, group);

            const controlCell = document.createElement('td');
            controlCell.setAttribute('colspan', '1');
            controlCell.classList.add('group-cell');
            controlCell.appendChild(button);

            const descCell = document.createElement('td');
            descCell.setAttribute('colspan', '12');

            const descContainer = document.createElement('div');
            descContainer.classList.add('grouping-row__desc');
            descCell.appendChild(descContainer);

            if (typeof content === 'string') {
                descContainer.textContent = content;
            } else {
                descContainer.appendChild(content);
            }

            if (currentPage > group.startPage) {
                const span = document.createElement('span');
                span.classList.add('grouping-row__info');
                span.textContent = 'продолжение группы';
                descContainer.appendChild(span);
            }

            const row = document.createElement('tr');
            row.classList.add('grouping-row');
            row.setAttribute('data-group-id', group.id);
            row.setAttribute('aria-expanded', isExpanded);
            row.appendChild(controlCell);
            row.appendChild(descCell);

            return row;
        };
    }

    RemoteGrouping.prototype.isEnable = isEnable;
    RemoteGrouping.prototype.setEnable = setEnable;
    RemoteGrouping.prototype.setDataSrc = setDataSrc;
    RemoteGrouping.prototype.getDataSrc = getDataSrc;
    RemoteGrouping.prototype.setGroups = setGroups;
    RemoteGrouping.prototype.getGroups = getGroups;
    RemoteGrouping.prototype.clear = clear;
    RemoteGrouping.prototype.getInfo = getInfo;

    function isEnable() {
        return this.enable;
    }

    function setEnable(flag) {
        this.enable = flag;
        
        if (flag === false && this.dataSrc.length > 0) {
            this.api.column(this.dataSrc + ':name').visible(true);
        }

        if (flag === true) {
            if (this.order !== undefined) {
                this.api.order(this.order);
            }

            const selector = this.dataSrc + ':name';

            // Change visibility after table refresh
            this.api.one('xhr.dt', function (e, settings) {
                const api = new $.fn.dataTable.Api(settings);
                api.column(selector).visible(false);
            });
        }
    }

    function setDataSrc(fieldName) {
        const oldDataSrc = this.dataSrc;

        this.dataSrc = fieldName;
        this.clear();

        if (this.enable) {
            // Change visibility after table refresh
            this.api.one('xhr.dt', function (e, settings) {
                const api = new $.fn.dataTable.Api(settings);

                if (oldDataSrc.length > 0)
                    api.column(oldDataSrc + ':name').visible(true);

                if (fieldName.length > 0)
                    api.column(fieldName + ':name').visible(false);
            });
        }

        const $table = this.api.tables().body().to$();
        $table.trigger('remote-grouping-datasrc', [this.api, this.dataSrc]);
    }

    function getDataSrc() {
        return this.dataSrc;
    }

    function setGroups(groupList) {
        const groups = [];
        groupList.forEach(function (value, index) {
            groups.push({
                id: index,
                name: value.Name,
                count: value.Count,
                description: value.Description,
                isExpanded: false
            });
        });

        this.groups = groups;
    }

    function getGroups() {
        return this.groups;
    }

    function clear() {
        this.groups = null;
    }
    
    function getInfo() {
        let recordsTotal = 0;
        this.groups.forEach(function (value, index) {
            recordsTotal += value.count;
        });

        return {
            recordsTotal: recordsTotal,
        };
    }

    return RemoteGrouping;
}());

(function () {
    $.fn.dataTable.Api.register('remoteGrouping()', function () {
        return this;
    });

    $.fn.dataTable.Api.register('remoteGrouping().enable()', function (flag) {
        flag = flag === undefined ? true : false;
        
        const ctx = this.context[0];

        if (ctx._remoteGrouping) {
            ctx._remoteGrouping.setEnable(flag);
        }

        return this;
    });

    $.fn.dataTable.Api.register('remoteGrouping().enabled()', function () {
        const ctx = this.context[0];

        if (!ctx._remoteGrouping) {
            return false;
        }

        return ctx._remoteGrouping.isEnable();
    });

    $.fn.dataTable.Api.register('remoteGrouping().dataSrc()', function (fieldName) {
        const ctx = this.context[0];

        if (!ctx._remoteGrouping) {
            return false;
        }

        if (fieldName !== undefined) {
            ctx._remoteGrouping.setDataSrc(fieldName);
        }

        return ctx._remoteGrouping.getDataSrc();
    });

    $.fn.dataTable.Api.register('remoteGrouping().groups()', function (groupList) {
        const ctx = this.context[0];

        if (!ctx._remoteGrouping) {
            return null;
        }

        if (groupList !== undefined) {
            ctx._remoteGrouping.setGroups(groupList);
        }

        return ctx._remoteGrouping.getGroups();
    });

    $.fn.dataTable.Api.register('remoteGrouping().setGroups()', function (groupList) {
        const ctx = this.context[0];

        if (!ctx._remoteGrouping) {
            return null;
        }

        if (groupList !== undefined) {
            ctx._remoteGrouping.setGroups(groupList);
        }

        return this;
    });

    $.fn.dataTable.Api.register('remoteGrouping().clear()', function () {
        const ctx = this.context[0];

        if (!ctx._remoteGrouping) {
            return null;
        }

        ctx._remoteGrouping.clear();

        return this;
    });

    $.fn.dataTable.Api.register('remoteGrouping().info()', function () {
        const ctx = this.context[0];

        if (!ctx._remoteGrouping) {
            return null;
        }

        return ctx._remoteGrouping.getInfo();
    });

    function _init(settings, options) {
        settings._remoteGrouping = new RemoteGrouping(settings, options);
    }

    // Entry point for initializing the extension
    $(document).on('preInit.dt', function (e, settings, json) {
        if (e.namespace !== 'dt') {
            return;
        }

        if (settings.oInit.remoteGrouping) {
            if (!settings._remoteGrouping) {
                _init(settings, settings.oInit.remoteGrouping);
            }
        }
    });
}());

var CustomAjaxStore = (function () {
    function CustomAjaxStore(options) {
        if (!options || !options.base || !options.base.url) return;

        this.isGrouping = false;

        this.base = {
            url: options.base.url,
            type: options.base.type || 'GET',
            datatype: options.base.datatype || 'json',
            dataSrc: options.base.dataSrc || 'data',
            cache: false
        }

        if (options.grouping) {
            if (!options.grouping.url) return;

            this.isGrouping = true;
            this.grouping = {
                url: options.grouping.url,
                type: options.grouping.type || 'GET',
                datatype: options.grouping.datatype || 'json',
                dataSrc: options.grouping.dataSrc || 'data',
                cache: false
            }
        }
    }

    CustomAjaxStore.prototype.getHandler = getHandler;

    function getHandler() {
        let that = this;

        return function (data, callback, settings) {
            const api = new $.fn.dataTable.Api(settings);
            let ajaxData = $.extend({}, that.base, { data: data });
            let ajaxDataGroups = $.extend({}, that.grouping, { data: data });

            if (api.remoteGrouping().enabled()) {
                if (api.remoteGrouping().groups() === null) {
                    $.when($.ajax(ajaxDataGroups), $.ajax(ajaxData)).then(successMulti, failure);
                } else {
                    $.ajax(ajaxData).then(success, failure);
                }
            } else {
                $.ajax(ajaxData).then(success, failure);
            }

            function success(rowsData, textStatus, jqXHR) {
                if (isRedirectToAuth(rowsData)) {
                    $.fn.dataTable.ext.errMode(settings, '', 'Error retrieving data from server');
                    api.processing(false);
                    return;
                }
                callback(rowsData);
            }

            function successMulti(groupListData, rowsData) {
                if (isRedirectToAuth(groupListData[0]) || isRedirectToAuth(rowsData[0])) {
                    $.fn.dataTable.ext.errMode(settings, '', 'Error retrieving data from server');
                    api.processing(false);
                    return;
                }

                if (groupListData[0][that.grouping.dataSrc] == null) {
                    $.fn.dataTable.ext.errMode(settings, '', 'В процессе выполнения произошла ошибка');
                    api.processing(false);
                    return;
                }

                const groups = groupListData[0][that.grouping.dataSrc];
                api.remoteGrouping().setGroups(groups);

                callback(rowsData[0]);
            }

            function failure(xhr, error, thrown) {
                $.fn.dataTable.ext.errMode(settings, '', thrown);
                api.processing(false);
            }
        };
    }

    return CustomAjaxStore;
}());