﻿var RowFocus = (function () {
    function RowFocus(settings, options) {
        this.api = new $.fn.dataTable.Api(settings);
        this.focusedRowId = null;

        this.c = $.extend({
            className: 'focused',
            enabled: true,
            disableOnRowSelection: false,
        }, options || {});

        this.api.on('draw.dtRowFocus', { context: this }, onDrawComplete);
        $(this.api.table().body()).on('click.dtRowFocus', 'tr', { context: this }, onClickRow);
        
        if (this.c.disableOnRowSelection) {
            this.api.on('select.dt', { context: this }, onSelectRows);
            this.api.on('deselect.dt', { context: this }, onDeselectRows);
        }
    }

    function onDrawComplete(e, settings) {
        const context = e.data.context;

        if (context.focusedRowId !== null && context.c.enabled) {
            const focusedRow = context.api.row(`#${context.focusedRowId}`);
            $(focusedRow.node()).addClass(context.c.className);
        }
    }

    function onClickRow(e) {
        const context = e.data.context;

        if (!context.c.enabled) {
            return;
        }

        const currentRow = context.api.row(this);
        const rowId = currentRow.id();

        _setFocusedRowId(context, rowId);
    }

    function onSelectRows(e, dt, type, indexes) {
        const context = e.data.context;

        if (context.api.rowFocus().enabled()) {
            _changeMode(context.api, 'select');
        }
    }

    function onDeselectRows(e, dt, type, indexes) {
        const context = e.data.context;

        const countRows = context.api.select.cumulative().rows.length;
        if (countRows === 0) {
            _changeMode(context.api, 'focus');
        }
    }

    function _setFocusedRowId(context, rowId) {
        const currentRow = context.api.row(`#${rowId}`);

        if (context.focusedRowId !== null) {
            const focusedRow = context.api.row(`#${context.focusedRowId}`);
            $(focusedRow.node()).removeClass(context.c.className);
        }

        $(currentRow.node()).addClass(context.c.className);
        context.focusedRowId = rowId;

        $(context.api.table().node()).trigger('dt-focus-change.dt', [rowId, currentRow.data()]);
    }
    
    function _changeMode(api, mode) {
        if (mode === 'focus') {
            api.rowFocus().enable(true);
            api.select.selector('td:first-child');
        } else if (mode === 'select') {
            api.rowFocus().enable(false);
            api.select.selector('tr');
        } else {
            console.error('Unsupported table mode');
        }
    }

    RowFocus.prototype.isEnable = isEnable;
    RowFocus.prototype.setEnable = setEnable;
    RowFocus.prototype.setFocusedRowId = setFocusedRowId;
    RowFocus.prototype.getFocusedRowId = getFocusedRowId;
    RowFocus.prototype.reset = reset;

    function isEnable() {
        return this.c.enabled;
    }

    function setEnable(flag) {
        this.c.enabled = flag;

        if (flag === false) {
            if (this.focusedRowId !== null) {
                const focusedRow = this.api.row(`#${this.focusedRowId}`);
                $(focusedRow.node()).removeClass(this.c.className);
            }

            this.focusedRowId = null;
        }
    }

    function getFocusedRowId() {
        const row = this.api.row(`#${this.focusedRowId}`);
        return row.any() ? this.focusedRowId : null;
    }

    function setFocusedRowId(rowId) {
        if (this.c.enabled && this.api.row(`#${rowId}`).any()) {
            _setFocusedRowId(this, rowId);
        }
    }

    function reset(){
        if (this.c.enabled) {
            if (this.focusedRowId !== null) {
                const focusedRow = this.api.row(`#${this.focusedRowId}`);
                $(focusedRow.node()).removeClass(this.c.className);
            }

            _changeMode(this.api, 'focus');
            this.focusedRowId = null;
        }
    }

    return RowFocus;
}());

(function () {   
    $.fn.dataTable.Api.register('rowFocus()', function () {
        return this;
    });

    $.fn.dataTable.Api.register('rowFocus().enable()', function (flag) {
        flag = flag === true;
        const context = this.context[0];

        if (context._rowFocus) {
            context._rowFocus.setEnable(flag);
        }

        return this;
    });

    $.fn.dataTable.Api.register('rowFocus().enabled()', function () {
        const context = this.context[0];

        if (!context._rowFocus) {
            return false;
        }

        return context._rowFocus.isEnable();
    });

    $.fn.dataTable.Api.register('rowFocus().focusedRowId()', function (rowId = null) {
        const context = this.context[0];

        if (!context._rowFocus || !context._rowFocus.isEnable()) {
            return null;
        }
        
        if (rowId !== null) {
            context._rowFocus.setFocusedRowId(rowId);
        }

        return context._rowFocus.getFocusedRowId();
    });

    $.fn.dataTable.Api.register('rowFocus().reset()', function () {
        const context = this.context[0];

        if (!context._rowFocus) {
            return false;
        }

        return context._rowFocus.reset();
    });

    function _init(settings) {
        const options = settings.oInit.rowFocus;
        settings._rowFocus = new RowFocus(settings, options);
    }

    // Entry point for initializing the extension
    $(document).on('preInit.dt', function (e, settings) {
        if (e.namespace !== 'dt') {
            return;
        }

        if (settings.oInit.rowFocus) {
            if (!settings._rowFocus) {
                _init(settings);
            }
        }
    });
}());