const LogConfigSettingsCollection = (function() {
    function LogConfigSettingsCollection() {
        this.values = [];
    }
    LogConfigSettingsCollection.prototype.push = function(logConfigSettings) {
        if (!logConfigSettings instanceof LogConfigSettings)
            throw new Error("Parameter \"logConfigSettings\" is not an instance of type LogConfigSettings.");

        this.values.push(logConfigSettings);
    }
    LogConfigSettingsCollection.prototype.pop = function(logConfigSettings) {
        if (!logConfigSettings instanceof LogConfigSettings)
            throw new Error("Parameter \"logConfigSettings\" is not an instance of type LogConfigSettings.");

        this.values.pop(logConfigSettings);
    }
    LogConfigSettingsCollection.prototype.filter = function(func) {
        this.values = this.values.filter(x => func(x));
        return this;
    }
    LogConfigSettingsCollection.prototype.map = function(func) {
        return this.values.map(x => func(x));
    }
    LogConfigSettingsCollection.prototype.hasElement = function(element) {
        if (!element instanceof LogConfigSettings)
            return false;

        for (let i = 0; i < this.values.length; i++)
            if (element.isEqualTo(this.values[i]))
                return true;

        return false;
    }
    LogConfigSettingsCollection.prototype.getById = function(id) {
        let results = this.values.filter(x => x.id === id);
        return results && results.length > 0 ? results[0] : null;
    }
    LogConfigSettingsCollection.prototype.stringify = function() {
        let request = $('#applySettingsForScanServices').is(':checked') === false 
            ? { "ConfigSettings": this.map(x => x.export()) }
            : { "ApplyToAll": true, "ConfigSettings": [selectedConfig.export()] };
        
        return JSON.stringify(request);
    }
    LogConfigSettingsCollection.prototype.getLength = function() {
        return this.values.length;
    }

    return LogConfigSettingsCollection;
})();

const LogConfigSettings = (function(id, type, logsPath, minLogLevel, storageDurationDays, syslogServer, syslogMinLogLevel, initialState) {
    function LogConfigSettings(data) {
        if (!data instanceof Object)
            throw new Error("Parameter \"data\" is not an instance of type Object.");

        this.id = data.Id;
        this.type = data.Type;
        this.logsPath = data.LogsPath;
        this.minLogLevel = data.MinLogLevel;
        this.storageDurationDays = data.StorageDurationDays;
        let syslogServerMatch = data.SyslogServer ? data.SyslogServer.match(syslogServerRegex) : null;
        this.syslogServer = syslogServerMatch ? syslogServerMatch[1] : null;
        this.syslogPort = syslogServerMatch ? syslogServerMatch[2] : 514;
        this.syslogMinLogLevel = data.SyslogMinLogLevel;
        
        this.initialState = {
            id: this.id,
            type: this.type,
            logsPath: this.logsPath,
            minLogLevel: this.minLogLevel,
            storageDurationDays: this.storageDurationDays,
            syslogServer: this.syslogServer,
            syslogPort: this.syslogPort,
            syslogMinLogLevel: this.syslogMinLogLevel
        };
    }
    LogConfigSettings.prototype.id = id;
    LogConfigSettings.prototype.type = type;
    LogConfigSettings.prototype.isEqualTo = function(other) {
        return other instanceof LogConfigSettings && this.id === other.id
    }

    LogConfigSettings.prototype.isValid = function() {
        const filePathRegex = /^(?:[a-zA-Z]:\\(?:[^\\/:*?"<>|\r\n]+\\)*[^\\/:*?"<>|\r\n]*|\/(?:[^\/\0]+\/)*[^\/\0]*)$/;
        if (filePathRegex.test(this.logsPath) !== true)
            return { result: false, message: "Необходимо указать корректный путь к файлу логов" };

        const positiveIntStrict = /^(?:0|[1-9][0-9]*)$/;
        if (positiveIntStrict.test(this.storageDurationDays) !== true)
            return { result: false, message: "Количество дней хранения логов должно быть положительным целым числом" };
        
        if (this.storageDurationDays > 2147483647)
            return { result: false, message: "Количество дней хранения логов не может превышать 2 147 483 647" };
        
        if (this.syslogServer) {
            if (syslogServerRegex.test(`udp://${this.syslogServer}:${this.syslogPort}`) !== true)
                return { result: false, message: "Адрес и порт сервера Syslog должны быть указаны корректно" };

            if (!this.syslogPort < 1 && this.syslogPort > 65535)
                return { result: false, message: "Значение порта сервера Syslog должно быть целым числом в диапазоне от 1 до 65535" };
            
            if (!this.syslogMinLogLevel)
                return { result: false, message: "Необходимо указать уровень логирования Syslog" };
        }

        return { result: true };
    }

    LogConfigSettings.prototype.isChanged = function() {
        return this.logsPath !== this.initialState.logsPath || this.minLogLevel !== this.initialState.minLogLevel || this.storageDurationDays !== this.initialState.storageDurationDays 
            || this.syslogServer !== this.initialState.syslogServer || this.syslogPort !== this.initialState.syslogPort || this.syslogMinLogLevel !== this.initialState.syslogMinLogLevel;
    }

    LogConfigSettings.prototype.export = function() {
        return {
            Id: this.id,
            Type: this.type,
            LogsPath: this.logsPath,
            MinLogLevel: this.minLogLevel,
            StorageDurationDays: this.storageDurationDays,
            SyslogServer: this.syslogServer ? `udp://${this.syslogServer}:${this.syslogPort}` : '',
            SyslogMinLogLevel: this.syslogServer ? this.syslogMinLogLevel : null
        };
    }

    return LogConfigSettings;
})();

function checkLoggingSettings() {
    if (!selectedConfig)
        return true;
    
    let isValid = selectedConfig.isValid();
    if (isValid.result !== true)
    {
        notifyWarning(isValid.message);
        return false;
    }
    
    return true;
}

function onBeforeSubmitLogging($form) {
    if (changedConfigs && changedConfigs.getLength() > 0)
        $form.append(
            $('<input>')
                .attr('type', 'hidden')
                .attr('name', 'Logging.ChangedLoggingSettings')
                .val(changedConfigs.stringify())
        );
}

function applyUserInput(propertyName, columnIndex, newValue, newDisplayValue) {
    if ($('#applySettingsForScanServices').is(':checked') !== true) {
        let selectedRows = table.rows('.selected');
        let rowIndex = selectedRows.indexes()[0];
        if (rowIndex >= 0) {
            table.cell(rowIndex, columnIndex).data(newDisplayValue ?? newValue);
        }
    } else {
        table.rows().every(function (rowIndex, tableLoop, rowLoop) {
           let data = this.data();
           if (data['Type'] === 1) { // Scan service
               table.cell(rowIndex, columnIndex).data(newDisplayValue ?? newValue);
           }
        });
    }

    if (selectedConfig) {
        selectedConfig[propertyName] = newValue;
        changedConfigs.filter(c => c.id !== selectedConfig.id);
        if (selectedConfig.isChanged() === true)
            changedConfigs.push(selectedConfig);
    }
}

let table = null;
let currentPageInitialData = null;
let changedConfigs = null;
let selectedConfig = null;
const syslogServerRegex = /^udp:\/\/([^:\/]+):(\d+)$/;

$(document).ready(function () {
    table = $('#LoggingSettingsGrid').DataTable({
        serverSide: true,
        processing: true,
        pageLength: 20,
        ajax: { url: '/LogConfigs/ComponentLogConfigsGrid' },
        select: {
            style: 'single',
            info: false
        },
        order: [[1, 'asc']],
        columns: [
            { data: 'Id', name: 'Id', visible: false },
            { data: 'ComponentName', name: 'ComponentName' },
            { data: 'Type', name: 'Type', render: function (data) {
                    let title;
                    let key;

                    switch (data) {
                        case 0:  // API
                            key = 'API';
                            title = "API";
                            break;
                        case 1:  // Scan service
                            key = 'ScanService';
                            title = "Служба сканирования";
                            break;
                        case 2:  // Sync service
                            key = 'SyncService';
                            title = "Служба синхронизации";
                            break;
                        case 3:  // Web client
                            key = 'Client';
                            title = "WEB клиент";
                            break;
                        default:
                            key = '';
                            title = '';
                    }

                    return '<span class="js-job-status" data-id="' + key + '">' + title + '</span>';
                } 
            },
            { data: 'LogsPath', name: 'LogsPath' },
            { data: 'MinLogLevel', name: 'MinLogLevel' },
            { data: 'StorageDurationDays', name: 'StorageDurationDays' },
            { data: 'SyslogServer', name: 'SyslogServer' },
            { data: 'SyslogMinLogLevel', name: 'SyslogMinLogLevel' }
        ],
    });
    
    changedConfigs = new LogConfigSettingsCollection();
    selectedConfig = null;

    table.on('select', function(e, dt, type, indexes) {
        if (type === 'row') {
            let data = table.row(indexes[0]).data();
            selectedConfig = changedConfigs.getById(data.Id) !== null ? changedConfigs.getById(data.Id) : new LogConfigSettings(data);
            
            $('#currentLogConfigLogsPath').val(selectedConfig.logsPath).parent().removeClass('disableContent');
            $('#currentLogConfigMinLogLevel').val(selectedConfig.minLogLevel).parent().removeClass('disableContent');
            $('#currentLogConfigStorageDurationDays').val(selectedConfig.storageDurationDays).parent().removeClass('disableContent');
            $('#currentLogConfigSyslogServer').val(selectedConfig.syslogServer).parent().removeClass('disableContent');
            if ($('#currentLogConfigSyslogServer').val() !== '' && $('#currentLogConfigSyslogServer').val() !== null) {
                $('#currentLogConfigSyslogPort').val(selectedConfig.syslogPort).parent().removeClass('disableContent');
                $('#currentLogConfigSyslogMinLogLevel').val(selectedConfig.syslogMinLogLevel).parent().removeClass('disableContent');
            }
            
            if (data['Type'] === 1) { // Scan service
                $('#applySettingsForScanServices').parent().parent().removeClass('disableContent');
            } else {
                $('#applySettingsForScanServices').parent().parent().addClass('disableContent');
            }
        }
    });

    table.on('user-select', function (e, dt, type, cell, originalEvent) {
        if (!selectedConfig)
            return;

        let isValid = selectedConfig.isValid();
        if (isValid.result !== true)
        {
            notifyWarning(isValid.message);
            e.preventDefault();
        }
    });

    table.on('draw', function() {
        if (!selectedConfig) {
            $('#currentLogConfigLogsPath').val('').parent().addClass('disableContent');
            $('#currentLogConfigMinLogLevel').val('').parent().addClass('disableContent');
            $('#currentLogConfigStorageDurationDays').val('').parent().addClass('disableContent');
            $('#currentLogConfigSyslogServer').val('').parent().addClass('disableContent');
            $('#currentLogConfigSyslogPort').val('').parent().addClass('disableContent');
            $('#currentLogConfigSyslogMinLogLevel').val('').parent().addClass('disableContent');
            $('#applySettingsForScanServices').parent().parent().addClass('disableContent');
        }
        
        currentPageInitialData = [];
        table.rows({ search: 'applied' }).every(function() {
            let rowIndex = this.index();
            let syslogServerMatch = this.data().SyslogServer ? this.data().SyslogServer.match(syslogServerRegex) : null;
            currentPageInitialData.push({
                Id: this.data().Id,
                Type: this.data().Type,
                LogsPath: this.data().LogsPath,
                MinLogLevel: this.data().MinLogLevel,
                StorageDurationDays: this.data().StorageDurationDays,
                SyslogServer: syslogServerMatch ? syslogServerMatch[1] : '',
                SyslogPort: syslogServerMatch ? syslogServerMatch[2] : '',
                SyslogMinLogLevel: this.data().SyslogMinLogLevel
            });
            
            let id = this.data().Id;
            let changedConfig = changedConfigs.getById(id);
            if (changedConfig === null)
                return;
            
            table.cell(rowIndex, 3).data(changedConfig.logsPath);
            table.cell(rowIndex, 4).data(changedConfig.minLogLevel);
            table.cell(rowIndex, 5).data(changedConfig.storageDurationDays);
            table.cell(rowIndex, 6).data(changedConfig.syslogServer ? `udp://${changedConfig.syslogServer}:${changedConfig.syslogPort}` : '');
            table.cell(rowIndex, 7).data(changedConfig.syslogMinLogLevel);

            if (selectedConfig && this.data().Id === selectedConfig.id)
                table.row(rowIndex).select();
        });

        if ($('#applySettingsForScanServices').is(':checked') === true) {
            applyUserInput('logsPath', 3, selectedConfig.logsPath);
            applyUserInput('minLogLevel', 4, selectedConfig.minLogLevel);
            applyUserInput('storageDurationDays', 5, selectedConfig.storageDurationDays);
            applyUserInput('syslogServer', 6, selectedConfig.syslogServer, selectedConfig.syslogServer ? `udp://${selectedConfig.syslogServer}:${selectedConfig.syslogPort}` : '');
            applyUserInput('syslogMinLogLevel', 7, selectedConfig.syslogMinLogLevel);
        }
    });

    $('#currentLogConfigLogsPath').on('keyup', function () {
        applyUserInput('logsPath', 3, $(this).val());
    });

    $('#currentLogConfigMinLogLevel').change(function () {
        applyUserInput('minLogLevel', 4, $(this).val());
    });

    $('#currentLogConfigStorageDurationDays').on('keyup', function () {
        applyUserInput('storageDurationDays', 5, $(this).val());
    });

    $('#currentLogConfigSyslogServer').on('keyup', function () {
        let displayValue = '';
        if ($(this).val() !== '')
            displayValue ='udp://' + $(this).val() + ':' + ($('#currentLogConfigSyslogPort').parent().hasClass('disableContent') === true ? selectedConfig.syslogPort : $('#currentLogConfigSyslogPort').val());
                
        applyUserInput('syslogServer', 6, $(this).val(), displayValue);
        if ($(this).val() === '') {
            $('#currentLogConfigSyslogPort').val('').parent().addClass('disableContent');
            $('#currentLogConfigSyslogMinLogLevel').val('').parent().addClass('disableContent');
            applyUserInput('syslogMinLogLevel', 7, '');
        } else {
            $('#currentLogConfigSyslogPort').val(selectedConfig.syslogPort).parent().removeClass('disableContent');
            $('#currentLogConfigSyslogMinLogLevel').val(selectedConfig.syslogMinLogLevel).parent().removeClass('disableContent');
            applyUserInput('syslogMinLogLevel', 7, selectedConfig.syslogMinLogLevel);
        }
    });

    $('#currentLogConfigSyslogPort').on('keyup', function () {
        applyUserInput('syslogPort', 6, $(this).val(), 'udp://' + $('#currentLogConfigSyslogServer').val() + ':' + $(this).val() );
    });

    $('#currentLogConfigSyslogMinLogLevel').change(function () {
        applyUserInput('syslogMinLogLevel', 7, $(this).val());
    });
    
    $('#applySettingsForScanServices').change(function() {
        if ($(this).is(':checked') === true) {
            $('#LoggingSettingsGridContainer').addClass('disableContent');
            applyUserInput('logsPath', 3, $('#currentLogConfigLogsPath').val());
            applyUserInput('minLogLevel', 4, $('#currentLogConfigMinLogLevel').val());
            applyUserInput('storageDurationDays', 5, $('#currentLogConfigStorageDurationDays').val());
            applyUserInput('syslogServer', 6, $('#currentLogConfigSyslogServer').val(), $('#currentLogConfigSyslogServer').val() ? `udp://${$('#currentLogConfigSyslogServer').val()}:${$('#currentLogConfigSyslogPort').val()}` : '');
            applyUserInput('syslogMinLogLevel', 7, $('#currentLogConfigSyslogMinLogLevel').val());
        } else {
            $('#LoggingSettingsGridContainer').removeClass('disableContent');
            table.rows({ search: 'applied' }).every(function() {
                let id = this.data().Id;
                let changedConfig = changedConfigs.getById(id);
                let rowIndex = this.index();
                if (changedConfig === null) {
                    let initialData = currentPageInitialData.filter(e => e.Id === id)[0];
                    table.cell(rowIndex, 3).data(initialData.LogsPath);
                    table.cell(rowIndex, 4).data(initialData.MinLogLevel);
                    table.cell(rowIndex, 5).data(initialData.StorageDurationDays);
                    table.cell(rowIndex, 6).data(initialData.SyslogServer ? `udp://${changedConfig.syslogServer}:${changedConfig.syslogPort}` : '');
                    table.cell(rowIndex, 7).data(initialData.SyslogMinLogLevel);
                } else {
                    table.cell(rowIndex, 3).data(changedConfig.logsPath);
                    table.cell(rowIndex, 4).data(changedConfig.minLogLevel);
                    table.cell(rowIndex, 5).data(changedConfig.storageDurationDays);
                    table.cell(rowIndex, 6).data(changedConfig.syslogServer ? `udp://${changedConfig.syslogServer}:${changedConfig.syslogPort}` : '');
                    table.cell(rowIndex, 7).data(changedConfig.syslogMinLogLevel);
                }
            });
        }
    });

    $('[data-toggle="tab"]').on('shown.bs.tab', function (e) {
        $.fn.dataTable.util.columnsAjustSize();
    });
});