import os, enum, io, re, json
from pypsrp.powershell import *
from pypsrp.client import Client
from pypsrp.complex_objects import GenericComplexObject

class TempTemplateType(enum.Enum):
    Default = 0
    User = 1
    System = 2


class TCredential:
    def __init__(self, name: str, password: str, domain: str = None, **kwargs):
        self.Name = name
        self.Password = password
        self.Domain = domain
        self.WinrmOverHttps = kwargs.get('winrmOverHttps')
        self.WinrmPort = kwargs.get('winrmPort')
        self.WinrmCertValidation = kwargs.get('winrmCertValidation') == 1
        self.Auth = kwargs.get('auth') #basic, certificate, negotiate, ntlm, kerberos, credssp(не поддерживается)

class WinrmTunnel:
    def __init__(self, host: str = "localhost", credential: TCredential = None):
        self._runspace: RunspacePool = None
        self._expandArchiveCommandExists: bool = True

        self.Host = host
        self.Credential = credential

    def __exit__(self):
        self.__Dispose()

    def __del__(self):
        self.__Dispose()

    def __Dispose(self):
        if self._runspace is not None or self._runspace:
            try:
                self._runspace.close()
            except:
                pass

    def Ping(self, **kwargs):  # => bool
        rs: RunspacePool = None

        try:
            # You can check params https://github.com/jborean93/pypsrp/blob/master/src/pypsrp/wsman.py#L144
            # TODO encapsulate all params here
            connectionInfo = Client(self.Host, ssl=self.Credential.WinrmOverHttps,
                                    port=self.Credential.WinrmPort,
                                    username=self.Credential.Name, password=self.Credential.Password,
                                    cert_validation=self.Credential.WinrmCertValidation,
                                    auth=self.Credential.Auth,
                                    **kwargs)

            rs = RunspacePool(connectionInfo.wsman, configuration_name=DEFAULT_CONFIGURATION_NAME)
            rs.open()
            return True
        except (Exception):
            return False
        finally:
            rs.close()

    def Connect(self, **kwargs):
        if self._runspace != None:
            raise Exception("Connection is already established.")

        if self.Credential is None:
            if self.Host != "localhost":
                raise Exception("InvalidOperationException: Credential can be empty only for localhost connection.")
            self._runspace = RunspacePool("localhost", configuration_name=DEFAULT_CONFIGURATION_NAME)
        else:
            # You can check params https://github.com/jborean93/pypsrp/blob/master/src/pypsrp/wsman.py#L144
            # TODO encapsulate all params here
            connectionInfo = Client(self.Host, ssl=self.Credential.WinrmOverHttps,
                                    port=self.Credential.WinrmPort,
                                    username=self.Credential.Name, password=self.Credential.Password,
                                    cert_validation=self.Credential.WinrmCertValidation,
                                    auth=self.Credential.Auth,
                                    **kwargs)
            self._runspace = RunspacePool(connectionInfo.wsman, configuration_name=DEFAULT_CONFIGURATION_NAME)

        self._runspace.open()

        self.__TryLogPsVersion()
        self.__VerifySessionMemoryQuota()

        # check Expand-Archive command
        self._expandArchiveCommandExists = self.CheckCommandExists("Expand-Archive")

        # throw exceptions for any troubles
        self.__SetVariable("ErrorActionPreference", "Stop")

    def Disconnect(self):
        if self._runspace is not None:
            self._runspace.close()
            self._runspace = None

    def IsPathExists(self, path: str, type_target: str = None):
        """Функция идентификации существования объекта в файловой системе по пути "path"
        если указать тип, то так же позволит узнать существование конкретного типа объекта
        type_target может быть Any(что угодно|по умолчанию),  Leaf(файл), Container(папка) 
        Args:
            path (str): Путь к интересующему объекту\n
            type_target (str): Тип объекта\n
        Returns:
            bool: Если существует, то "True" иначе, "False"
        """
        params = { "Path": path.rstrip(os.sep) }
        if type_target is not None:
            params["PathType"] = type_target
        response = self.__ExecuteShellCommand("Test-Path", params)
        return WinrmTunnel.__OutputExecuteShell(response)[0] == "True"

    def Upload(self, fileOrFolderPath: str, destinationFilePath : str = "", bufferSize: int = 1024, isRecursively: bool = True, isOnlyContentFolder = False):
        destinationFilePath = WinrmTunnel.__CheckCorrectPath(destinationFilePath)
        if (os.path.isdir(fileOrFolderPath)):
            if isOnlyContentFolder:
                targetDir = destinationFilePath
            else:
                targetDir = destinationFilePath + "\\" + os.path.basename(fileOrFolderPath)
            self.__UploadFolder(fileOrFolderPath, targetDir, isRecursively, bufferSize)
        else:
            self.__UploadFile(fileOrFolderPath, destinationFilePath, bufferSize)

    def UploadFile(self, filePath: str, destinationFilename: str, bufferSize: int = 1024):
        self.__UploadFile(filePath, WinrmTunnel.__CheckCorrectPath(destinationFilename), bufferSize)

    def UploadFileEx(self, filePath: str, destinationFilePath: str, bufferSize: int = 1024):
        self.__UploadFile(filePath, WinrmTunnel.__CheckCorrectPath(destinationFilePath), bufferSize)

    def Download(self, filePath: str, destinationPath: str, bufferSize: int = 1024, isOverwrite: bool = False, isOnlyContentFolder = False):
        if self.IsPathExists(filePath, "Container"):
            if not os.path.isdir(destinationPath):
                os.makedirs(destinationPath)
            listObjects : list[GenericComplexObject] = self.__ExecuteShellCommand("Get-Childitem", { "Path" : filePath , "Name" : None })
            for i in range(len(listObjects)):
                target : GenericComplexObject = listObjects[i]
                target_str : str = target.property_sets[0]
                if isOnlyContentFolder:
                    self.Download("\\".join([filePath, target_str]), os.sep.join([destinationPath, target_str]),
                                  bufferSize, isOverwrite, True)
                else:
                    name_folder = filePath.split("\\")[-1]
                    os.mkdir(os.sep.join([destinationPath, name_folder]))
                    self.Download("\\".join([filePath, target_str]), os.sep.join([destinationPath, name_folder, target_str]),
                                  bufferSize, isOverwrite, True)
        elif self.IsPathExists(filePath, "Leaf"):
            self.DownloadFile(filePath, destinationPath, bufferSize, isOverwrite)
        else:
            print(f"py: Файл/папка по пути \"{filePath}\" не найдены!")

    def DownloadFile(self, filePath: str, destinationPath: str, bufferSize: int = 1024, isOverwrite: bool = False):
        filePath = self.GetFullPath(filePath)
        print(f"py: Download file '{filePath}' '{destinationPath}'..")
        if os.path.isfile(destinationPath):
            if isOverwrite:
                os.remove(destinationPath)
            else:
                print(f"py: Download. Returning due to os.path.exists condition")
                return

        self.__ExecuteShellScript(["$altxStream = [IO.File]::OpenRead('" + filePath + "')"])

        try:
            self.__SetVariable("altxBufferSize", bufferSize)
            self.__ExecuteShellScript(["$altxBuffer = new-object Byte[] $altxBufferSize"])
            n = 0
            with io.open(destinationPath, 'wb', bufferSize) as writer:
                bytesRead: int = None
                while True:
                    self.__ExecuteShellScript(["$altxBytesRead = $altxStream.Read($altxBuffer, 0, $altxBufferSize)"])

                    try:
                        outputExecuteShellStr = WinrmTunnel.__OutputExecuteShell(self.__ExecuteShellScript(["$altxBytesRead"]))[0]
                        bytesRead = int(outputExecuteShellStr)

                        if bytesRead > 0:
                            result = self.__ExecuteShellScript(["Get-Variable -Name altxBuffer -ValueOnly"])

                            buffer: bytes = bytes(result[0][0:bytesRead])

                            writer.write(buffer)  # .Write(buffer, 0, bytesRead)
                            n = n + 1
                        else:
                            break
                    except Exception as err:
                        print(f"py: ERROR {type(err)}: outputExecuteShellStr => {{{str(outputExecuteShellStr)}}}")
                        print(f"py: Ошибка чтения файла. Возможно файла \"{filePath}\" не существует!")
                        raise err
                    finally:
                        writer.flush()
        finally:
            script = ["$altxStream.Close()",
                      "$altxStream.Dispose()",
                      "Remove-Variable altx*"]

            self.__ExecuteShellScript(script)

    def Execute(self, command_Type: str, command: str, parameters: dict = None, **kwargs):
        """Функция вызова удалённых команд

        Args:
            command_Type (str): "cmdlet" или "executable"\n
            command (str): Команда\n
            parameters (dict[str, any]): параметры команды со значениями {param1 : val1, param2 : val2, ...}\n

        Returns:
            list[ComplexObject]: Результат выполненния, если его нету, то возвращает "None"
        """

        print(f"py: Running '{command_Type}' '{command}' {parameters} kwargs={kwargs}..")

        command_Type = command_Type.lower()
        if (command_Type != "executable" and command_Type != "cmdlet"):
            raise Exception(f"InvalidOperationException( Unsupported tunnel command type: {command_Type})")

        # Replace Expand-Archive command with a script based on shell.application if PowerShell is less than 5.0
        # https://stackoverflow.com/questions/37814037/how-to-unzip-a-zip-file-with-powershell-version-2-0
        if (command_Type == "cmdlet" and command.lower() == "expand-archive"
                and not self._expandArchiveCommandExists):
            path = parameters.get("Path", "`")
            if not (path != "`" and isinstance(path, str)):
                raise Exception(
                    f"ArgumentException(Parameter \"Path\" is not specified or invalid for \"Expand-Archive\" command.)")
            destinationPath = parameters.get("DestinationPath", "`")
            if not (destinationPath != "`") and isinstance(destinationPath, str):
                raise Exception(
                    f"ArgumentException(Parameter \"DestinationPath\" is not specified or invalid for \"Expand-Archive\" command.")
            self.__ExecuteShellScript(WinrmTunnel.__CreateExpandZipScriptViaShell(path, destinationPath))
            print(f"py: _expandArchiveCommandExists exit")
            return None

        if command_Type == "executable":
            return self.__RunExecutable(command, parameters, **kwargs)

        elif command_Type == "cmdlet" and (parameters is None or isinstance(parameters, dict)):
            result = self.__ExecuteShellCommand(command, parameters)
            if len(result) == 1:
                return json.dumps(result, default=lambda o: o.__dict__)
            else:
                return None
    
    def CheckProcessExists(self, PidProcess : str):
        check = self.__ExecuteShellCommand("Get-Process", { "Id" : PidProcess, "ErrorAction" : "SilentlyContinue" })
        if check:
            return True
        else:
            return False

    def CheckCommandExists(self, commandName: str):  # => bool
        parameters = {"Name": commandName, "ErrorAction": "SilentlyContinue"}
        return len(self.__ExecuteShellCommand("Get-Command", parameters)) > 0

    def RemovePath(self, path: str, isRecurse: bool):
        """Процедура, которая удаляет файл/папку лежащий на пути path
            Данная процедура не может удалить папку, путь к которой равен пути в котором происходит работа контекста PowerShell (path != self.GetCurrentDirectory())
        Args:
            path (str): Путь указывающий на удаляемый файл/папку\n
            isRecurse (bool): Удаление будит рекурсивным?\n
        """
        if path is None or path == "":
            raise Exception(f"ArgumentNullException({type(path)}")

        parameters = ({"Path": path}, {"Path": path, "Recurse": None},)[isRecurse]
        self.__ExecuteShellCommand("Remove-Item", parameters)

    def GetCurrentDirectory(self):
        """Функция, которая возвращает актуальный путь для рабочего runtime
        Returns:
            str: Актуальный путь для рабочего runtime
        """
        return str(self.__ExecuteShellCommand("Get-Location")[0])

    def GetFullPath(self, path : str):
        """Функция, которая возвращает полный путь в контексте выполнения рабочего runtime
        Args:
            path (str): Путь который нужно преобразовать\n
            parameters (_type_, optional): Параметры с их значениями, вызываемой команды\n
        Returns:
            str: Полный путь
        """
        workDir = self.GetCurrentDirectory()

        if path is None or path == "":
            return workDir
        else:
            if path[-1] == "\\":
                path = path[0:-1]
            if not re.fullmatch(r"(^.:\\.*)", path):
                workDir += "\\"
                if path.startswith(".\\"):
                    path = path.replace(".\\", workDir, 1)
                elif path.startswith("\\"):
                    path = path.replace("\\", workDir, 1)
                else:
                    path = workDir + path
                return path
            else:
                return path

    def CreateFolder(self, path: str):
        folderPath = path.rstrip(os.sep)
        parameters = {"Type": "Directory", "Path": folderPath}
        self.__ExecuteShellCommand("New-Item", parameters)

    def SetCurrentDirectory(self, path: str):
        self.__ExecuteShellCommand("Set-Location", {"Path": path})

    def GetTempPathByTemplate(self, template: TempTemplateType):
        if template == TempTemplateType.Default:
            command = "[System.IO.Path]::GetTempPath()"
        elif template == TempTemplateType.User:
            command = "[environment]::GetEnvironmentVariable(\"TEMP\",\"User\")"
        elif template == TempTemplateType.System:
            command = "[environment]::GetEnvironmentVariable(\"TEMP\",\"Machine\")"
        else:
            raise Exception(f"ArgumentOutOfRangeException: {type(template)} {str(template)}")

        response = self.__ExecuteShellScript([command]);
        return str(WinrmTunnel.__OutputExecuteShell(response)[0])

    def __TryLogPsVersion(self):
        try:
            psVersion = self._runspace.ps_version
            # Logger.Debug(f"PowerShell version {psVersion} is installed on \"{self.Host}\".")
        except Exception as ex:
            pass
        # Logger.Debug(ex, f"Unable to get PowerShell version on \"{self.Host}\".")

    def __VerifySessionMemoryQuota(self):
        MINIMUM_MEMORY_MB: int = 2048
        MEMORY_QUOTA: str = "MaxMemoryPerShellMB"
        MEMORY_QUOTA_PATH: str = r"WSMan:localhost\Shell\MaxMemoryPerShellMB"
        WSAN_CLIENT_REGISTRY_KEY: str = r"HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\WSMAN\Client"

        # At first try to get memory quota by PowerShell command
        # If it does not exist, then get the value from registry

        # ! Нужно реализовать функции __GetIntegerItem и __GetItemProperty позже
        # collectedMemoryQuotaMb = self.__GetIntegerItem(MEMORY_QUOTA_PATH) ?? self.__GetItemProperty<int?>(wsanClientRegistryKey, memoryQuota);
        collectedMemoryQuotaMb = MINIMUM_MEMORY_MB

        # If the value is not set, WinRM uses default value and I consider then it always less than target
        if collectedMemoryQuotaMb is None:
            raise Exception(
                f"InvalidOperationException: Can not get {MEMORY_QUOTA} actual quota value to verify memory limit. " +
                f"Try to set the quota through command `Set-Item {MEMORY_QUOTA_PATH} {MINIMUM_MEMORY_MB}` and restart WinRM service on remote machine.")
        # How configure memory limit
        # https://devblogs.microsoft.com/scripting/learn-how-to-configure-powershell-memory/
        elif collectedMemoryQuotaMb < MINIMUM_MEMORY_MB:
            raise Exception(
                f"InvalidOperationException: {MEMORY_QUOTA} quota of \"{self.Host}\" must be greater than or equal {MINIMUM_MEMORY_MB} MB. " +
                f"Run command `Set-Item {MEMORY_QUOTA_PATH} {MINIMUM_MEMORY_MB}` and restart WinRM service on remote machine.");

    def __RunExecutable(self, executablePath: str, arguments, **kwargs):
        if isinstance(arguments, list):
            in_list = arguments
            arguments = {}
            for i in in_list:
                arguments[i] = None

                # When working in the current directory, the "Start-Process" requires availability ".\" at the beginning of the FilePath
        executablePath = WinrmTunnel.__CheckCorrectPath(executablePath)
        # Escape spaces with quotes in arguments
        argumentList = []
        if arguments is None:
            parameters = {"FilePath": executablePath,
                          "NoNewWindow": None,
                          "LoadUserProfile": None,
                          "PassThru": None}
        else:
            for j in arguments.keys():
                for i in [j, arguments[j]]:
                    if i is not None:
                        if i.isspace():
                            argumentList.append(f"\"{i}\"")
                        else:
                            argumentList.append(i)

            parameters = {"FilePath": executablePath,
                          "ArgumentList": argumentList,
                          "NoNewWindow": None,
                          "LoadUserProfile": None,
                          "PassThru": None }

        isWait = (False, True)["Wait" in kwargs and kwargs["Wait"] == True]
        if isWait:
            parameters["Wait"] = None

        result = self.__ExecuteShellCommand("Start-Process", parameters)

        if result is not None and len(result) > 0:
            if (result[0]._to_string is not None and result[0]._to_string != ""):
                raise Exception(f"{result[0]._to_string} path:{executablePath}")
            return result[0].adapted_properties["Id"]
        else:
            return None

    def __ExecuteShellCommand(self, command: str, parameters: dict = None):
        """Функция, обращающаяся к удалённой машине через runspace, с командой 'command' и параметрами 'parameters'\n
        Записи происходят на прмере следующих шаблонов:\n
        Test-Command -param1 value1 -param2 value2\n
        или\n
        Test-Command -param1 -param2 -param3 -param4 value4 -param5 -param6 value6\n
        

        Args:
            runspace (RunspacePool)
            command (str): Вызываемая команда\n
            parameters (_type_, optional): Параметры с их значениями, вызываемой команды\n

        Returns:
            list[ComplexObject]
        """

        shell = PowerShell(self._runspace)
        shell.add_cmdlet(command)

        if parameters is not None:
            for parameter, val in parameters.items():
                if val == None:
                    shell.add_parameter(parameter_name=parameter)
                else:
                    shell.add_parameter(parameter_name=parameter, value=val)

        result = shell.invoke()

        if len(result) == 0 and len(shell.streams.error) > 0:
            raise Exception(f"{shell.streams.error[0]._to_string}. Command: {command}")
        return result  # list[ComplexObject]

    def __ExecuteShellScript(self, lines: list):
        shell = PowerShell(self._runspace)
        shell: PowerShell
        for line in lines:
            shell.add_script(line)
        result = shell.invoke()
        return result  # list[ComplexObject]

    def __SetVariable(self, name: str, value):
        self.__ExecuteShellCommand("Set-Variable", {"Name": name, "Value": value})

    def __UploadFolder(self, folderPath: str, baseFolderPath: str = "", recursively: bool = False, bufferSize: int = 1024):
        if not self.IsPathExists(baseFolderPath, "Container"):
            self.CreateFolder(baseFolderPath)
        for target in os.listdir(folderPath):
            targetPath = folderPath + os.sep + target
            targetWinPath = baseFolderPath + "\\" + target
            if os.path.isfile(targetPath):
                self.__UploadFile(targetPath, targetWinPath, bufferSize)
            elif recursively and os.path.isdir(targetPath):
                self.__UploadFolder(targetPath, targetWinPath, True, bufferSize)

    def __UploadFile(self, filePath: str, baseFolderPath: str = "", bufferSize: int = 1024):
        filename = os.path.basename(filePath)
        if filename is None or filename == "":
            raise Exception(f"Unable to get file name from \"{filePath}\".")

        destinationPath = self.GetFullPath(baseFolderPath)

        self.__UploadStream(filePath, destinationPath, bufferSize)

    def __UploadStream(self, filePath: str, destinationPath: str, bufferSize: int = 1024):
        if filePath is None or filePath == "":
            raise Exception(f"ArgumentNullException({type(filePath)})")

        if destinationPath is None or destinationPath == "":
            raise Exception(f"ArgumentNullException({type(destinationPath)})")

        print(f"py: UploadStream '{filePath}' '{destinationPath}'..")

        workDir = self.GetCurrentDirectory()
        destinationPath = self.GetFullPath(destinationPath)

        # overwrite destination file

        if self.IsPathExists(destinationPath):
            self.RemovePath(destinationPath, True)

        self.__ExecuteShellScript(["$altxStream = [IO.File]::OpenWrite('" + destinationPath + "')"])

        try:
            reader = io.open(filePath, 'rb', bufferSize)
            buffer: bytes = None
            while True:
                buffer = reader.read(bufferSize)
                if len(buffer) > 0:
                    self.__SetVariable("altxBuffer", buffer)
                    self.__ExecuteShellScript(["$altxStream.Write($altxBuffer, 0, $altxBuffer.Length)"])
                    self.__ExecuteShellCommand("Remove-Variable", {"Name": "altxBuffer"})
                else:
                    break
        finally:
            script = ["$altxStream.Close()",
                      "$altxStream.Dispose()",
                      "Remove-Variable altx*"]

            self.__ExecuteShellScript(script)
            # reader.position = 0;     

    def __CreateExpandZipScriptViaShell(zipPath: str, destinationFolderPath: str):
        # copyhere() options
        # https://docs.microsoft.com/en-us/windows/win32/shell/folder-copyhere

        # 1556
        # (0110 0001 0100)

        # (0000 0000 0100) Do not display a progress dialog box.
        # (0000 0001 0000) Respond with "Yes to All" for any dialog box that is displayed.
        # (0010 0000 0000) Do not confirm the creation of a new directory if the operation requires one to be created.
        # (0100 0000 0000) Do not display a user interface if an error occurs.

        return [f"$shell = New-Object -ComObject Shell.Application",
                f"$zip = $shell.NameSpace(\"{zipPath}\")",
                f"New-Item -ItemType Directory -Force -Path \"{destinationFolderPath}\"",
                f"foreach ($item in $zip.items()){{ $shell.Namespace(\"{destinationFolderPath}\").copyhere($item, 1556) }}"]

    def __CheckCorrectPath(path : str):
        """Функция преобразования пути в случаи несоответствия его вида для корректной работы
        Args:
            path (str): путь который нужно преобразовать\n
        Returns:
            str: Преобразованный путь
        """

        if not(path.startswith(".\\") or re.fullmatch(r"(^.:\\.*)", path)):
            if path.startswith("\\"):
                return "." + path
            else:
                return ".\\" + path
        return path

    def __OutputExecuteShell(exe_result: list):
        def fun_1(obj):
            result = []
            if isinstance(obj, list):
                result = fun_2(obj)
            elif isinstance(obj, GenericComplexObject) or isinstance(obj, ComplexObject):
                obj: GenericComplexObject
                for name in obj.adapted_properties:
                    result.append(f"{name} : {obj.adapted_properties[name]}")
            else:
                result.append(str(obj))
            return result

        def fun_2(mass_obj: list):
            result = []
            for obj in mass_obj:
                result = result + fun_1(obj)
            return result

        return fun_1(exe_result)