local shortport = require "shortport"
local sslcert = require "sslcert"
local sslv2 = require "sslv2"
local stdnse = require "stdnse"

description = [[
Determines whether the server supports obsolete and less secure SSLv2, and discovers which ciphers it
supports.
]]

---
--@output
-- 443/tcp open   https   syn-ack
-- | sslv2:
-- |   SSLv2 supported
-- |   ciphers:
-- |     SSL2_DES_192_EDE3_CBC_WITH_MD5
-- |     SSL2_IDEA_128_CBC_WITH_MD5
-- |     SSL2_RC2_128_CBC_WITH_MD5
-- |     SSL2_RC4_128_WITH_MD5
-- |     SSL2_DES_64_CBC_WITH_MD5
-- |     SSL2_RC2_128_CBC_EXPORT40_WITH_MD5
-- |_    SSL2_RC4_128_EXPORT40_WITH_MD5
--@xmloutput
--<elem>SSLv2 supported</elem>
--<table key="ciphers">
--  <elem>SSL2_DES_192_EDE3_CBC_WITH_MD5</elem>
--  <elem>SSL2_IDEA_128_CBC_WITH_MD5</elem>
--  <elem>SSL2_RC2_128_CBC_WITH_MD5</elem>
--  <elem>SSL2_RC4_128_WITH_MD5</elem>
--  <elem>SSL2_DES_64_CBC_WITH_MD5</elem>
--  <elem>SSL2_RC2_128_CBC_EXPORT40_WITH_MD5</elem>
--  <elem>SSL2_RC4_128_EXPORT40_WITH_MD5</elem>
--</table>


author = {"Matthew Boyle", "Daniel Miller"}
license = "Same as Nmap--See https://nmap.org/book/man-legal.html"

categories = {"default", "safe"}

	
local use_portrule = stdnse.get_script_args("stdnse.use_portrule") or false

if (use_portrule == "true") then
	portrule = function(host,port) 
			return shortport.ssl(host, port) or sslcert.getPrepareTLSWithoutReconnect(port) 
		end
else portrule = function(host,port) return true end
end


action = function(host, port)
  local ciphers = sslv2.test_sslv2(host, port)

  if ciphers then
    host.registry.sslv2 = host.registry.sslv2 or {}
    host.registry.sslv2[port.number .. port.protocol] = ciphers
    return {
      "SSLv2 supported",
      ciphers = #ciphers > 0 and ciphers or "none"
    }
  end
end
