﻿var ColumnResize = (function () {
    const DataTable = $.fn.dataTable;

    let _tableHeader;
    let _tableBody;
    let _oldWidth;
    let _oldWidthNext;
    let _isSimpleModeResize;
    let _tableSettings;

    function ColumnResize(settings) {
        _tableSettings = settings;

        _setUp();
    }

    function _setUp() {
        const api = new DataTable.Api(_tableSettings);

        _addTableWrapper(api.table().container());
        _tableHeader = $(api.table().header()).parent();
        _tableBody = $(api.table().body()).parent();

        api.on('init', function () {
            _tableSettings.oFeatures.bAutoWidth = false;
        });
        
        api.on('draw', function () {
            // It is necessary to add vertical scrolling 
            // so that the table width does not change
            const scrollBody = _tableBody.parent();
            scrollBody.css('overflow-y', 'scroll');

            _addResize(api);
        });

        let columnCount = _getNumberVisibleColumns(api);
        api.on('column-visibility.dt', function (e, settings, column, state) {
            // Check if the number of columns has changed
            const oldColumnCount = columnCount;
            columnCount = _getNumberVisibleColumns(api);
            if (columnCount === oldColumnCount) {
                return;
            }

            _tableSettings.aoColumns.forEach(function (column) {
                if (column.width === undefined) {
                    column.sWidth = null;
                }
            });
        });

        api.on('column-sizing.dt', function (e, settings) {
            api.columns().indexes().each(function (index) {
                const thBody = _getThForScrollBody(api, index);
                if (thBody) {
                    const thHeader = api.column(index).header();
                    const currentWidth = $(thHeader).outerWidth();

                    $(thBody).width(currentWidth);

                    const columnOptions = _tableSettings.aoColumns[index];
                    _checkNeedAdjust(columnOptions, currentWidth, function (isNeedAdjust, newWidth) {
                        if (isNeedAdjust) {
                            columnOptions.sWidth = newWidth + 'px';
                            $(thHeader).width(newWidth);
                            $(thBody).width(newWidth);
                        }
                    });
                }
            });

            const tableHeadWidth = _tableHeader.width();
            _tableHeader.parent().width(tableHeadWidth);
        });

        _tableHeader.on('resize', function (e) {
            e.stopPropagation();
        });

        const tableId = api.table().table().node().id;
        $(window).on('resize.DT-' + tableId, $.fn.dataTable.util.throttle(
            function () {
                api.columns.adjust();
            },
            250
        ));
    }

    function _addTableWrapper(container) {
        const wrapper = document.createElement('div');

        wrapper.classList.add('table-wrapper');
        wrapper.style.height = container.clientHeight + 'px';

        $(container).after(wrapper);
        wrapper.appendChild(container);
    }

    function _addResize(api) {
        const container = api.table().container();

        $(container).find('th').resizable({
            handles: 'e',
            cancel: 'th:last-child, .resize-none',
            minWidth: 15,
            resize: function (event, ui) {
                _editColumnWidth(api, ui.element, ui.size.width);
            },
            start: function (event, ui) {
                if (!ui.element.next().length) {
                    ui.element.trigger('mouseup');
                    return;
                }

                _isSimpleModeResize = _isHorizontalScrollTable(api);

                if (_isSimpleModeResize === false) {
                    const maxWidth = ui.element.width() + ui.element.next().width() - 15;
                    ui.element.resizable('option', 'maxWidth', maxWidth);
                }

                _startEditColumnWidth(api, ui.element, ui.size.width);
            },
            stop: function (event, ui) {
                _stopEditColumnWidth(api, ui.element, ui.size.width);
            },
        });
    }

    function _startEditColumnWidth(api, element, width) {
        _oldWidth = element.outerWidth();
        _oldWidthNext = element.next().outerWidth();

        if (_isSimpleModeResize) {
            // So that there are no graphic distortions and 
            // the columns take up the entire width of the table
            const lastOfHead = _tableHeader.find('col').last();
            lastOfHead.css('width', 'auto');
            lastOfHead.css('min-width', '');

            const lastOfBody = _tableBody.find('col').last();
            lastOfBody.css('width', 'auto');
            lastOfBody.css('min-width', '');
        }

        api.table().container().classList.add('is-column-resizing');
    }
    
    function _editColumnWidth(api, element, width) {
        if (_isSimpleModeResize) {
            _setWidth(api, element, width);
            return;
        }

        const difference = width - _oldWidth;
        const nextElement = element.next();

        _setWidth(api, nextElement, _oldWidthNext - difference);
        _setWidth(api, element, width);
    }

    function _stopEditColumnWidth(api, element, width) {
        api.table().container().classList.remove('is-column-resizing');

        api.columns.adjust();
    }

    function _setWidth(api, node, width) {
        const domIndex = node.index();
        
        // update column width in header
        _tableHeader.find('col').eq(domIndex).width(width);
        _tableHeader.find('col').eq(domIndex).css('min-width', width);

        // update column width in body
        _tableBody.find('col').eq(domIndex).width(width);
        _tableBody.find('col').eq(domIndex).css('min-width', width);

        const columnIndex = api.column(node).index();
        _tableSettings.aoColumns[columnIndex].sWidth = width + 'px';
    }

    function _getNumberVisibleColumns(api) {
        let column = api.columns().visible().toArray();

        column = column.filter(function (isVivible) {
            return isVivible;
        });

        return column.length;
    }

    function _isHorizontalScrollTable(api) {
        const tableWidth = api.table().body().clientWidth;
        const scrollBody = _tableBody.parent().get(0);

        return tableWidth > scrollBody.clientWidth;
    }

    function _getThForScrollBody(api, columnIndex) {
        const column = api.column(columnIndex);

        if (!column.visible()) {
            return;
        }

        const $thHeader = $(column.header());
        const thBody = _tableBody.find('col').eq($thHeader.index()).get(0);

        return thBody;
    }

    function _checkNeedAdjust(columnOptions, currentWidth, callback) {
        const width = columnOptions.sWidth;
        const minWidth = widthToInt(columnOptions.minWidth);

        if (width == null || width.indexOf('%') !== -1) {
            if (minWidth && currentWidth < minWidth) {
                return callback(true, minWidth);
            }
        }

        // DataTables using 'table-layout: fixed;' can collapse columns without given width
        if (currentWidth < 15) {
            return callback(true, 100);
        }

        return callback(false);
    }

    function widthToInt(widthString) {
        if (widthString) {
            return Number(widthString.replace('px', ''));
        }
        return 0;
    }

    return ColumnResize;
}());

(function () {
    function _init(settings) {
        settings._columnResize = new ColumnResize(settings);
    }

    // Entry point for initializing the extension
    $(document).on('preInit.dt', function (e, settings, json) {
        if (e.namespace !== 'dt') {
            return;
        }

        if (settings.oInit.columnResize) {
            if (!settings._columnResize) {
                _init(settings);
            }
        }
    });
}());