﻿/**
 * @typedef {Object} TableToolbarOptions
 * @property {array} contentLeft - Array of buttons for the left side.
 * @property {array} singleActions - Array of buttons for one row.
 * @property {array} batchActions - Array of buttons for multiple rows.
 * @property {array} contentRight - Array of buttons for the right side.
 */

class TableToolbar {
    /**
     * @param {DataTable.settings} settings
     * @param {TableToolbarOptions} options
     */
    constructor(settings, options) {
        this.api = new DataTable.Api(settings);
        this.config = options || {};

        this.strategies = {
            noSelection: {
                getConfig: () => ({
                    contentLeft: this.config.contentLeft || [],
                    contentRight: this.config.contentRight || []
                })
            },

            singleSelection: {
                getConfig: () => ({
                    contentLeft: this.config.singleActions || this.config.batchActions || this.config.contentLeft || [],
                    contentRight: this.config.contentRight || []
                })
            },

            multiSelection: {
                getConfig: () => ({
                    contentLeft: this.config.batchActions || [],
                    contentRight: this.config.contentRight || []
                })
            }
        };

        this.currentStrategy = this.strategies.noSelection;

        this.api.on('draw.dt select.dt deselect.dt dt-focus-change.dt', () => this._updateState());

        this._render();
    }

    async _getConfig() {
        const config = this.currentStrategy.getConfig();

        return {
            contentLeft: typeof config.contentLeft === 'function' ? await config.contentLeft(this.api) : config.contentLeft,
            contentRight: typeof config.contentRight === 'function' ? await config.contentRight(this.api) : config.contentRight,
        };
    }

    _updateState() {
        let selectedRows = this.api.redcheck.selectedIds();

        if (selectedRows.length === 0 && this.api.rowFocus().focusedRowId() !== null) {
            selectedRows = [this.api.rowFocus().focusedRowId()];
        }

        if (selectedRows.length === 1) {
            this.currentStrategy = this.strategies.singleSelection;
        } else if (selectedRows.length > 1) {
            this.currentStrategy = this.strategies.multiSelection;
        } else {
            this.currentStrategy = this.strategies.noSelection;
        }

        this._render();
    }

    async _render() {
        const config = await this._getConfig();
        this.clear();

        const contentLeft = new DataTable.Buttons(this.api, { buttons: config.contentLeft });
        const contentRight = new DataTable.Buttons(this.api, { buttons: config.contentRight });

        let tableToolbar = $('.table-toolbar', this.api.table().container());
        if (tableToolbar.length === 0) {
            $('<div></div>').addClass('table-toolbar').prependTo(this.api.table().container());
            tableToolbar = $('.table-toolbar', this.api.table().container());

            tableToolbar.append($('<div></div>').addClass('table-toolbar__left'));
            tableToolbar.append($('<div></div>').addClass('table-toolbar__right'));
        }

        contentLeft.container().appendTo(tableToolbar.find('.table-toolbar__left'));
        contentRight.container().appendTo(tableToolbar.find('.table-toolbar__right'));
    }

    clear() {
        for (const button of Array.from(this.api.buttons().nodes()).reverse()) {
            this.api.button(button).remove();
        }
        this.api.buttons().remove();

        const tableToolbar = $('.table-toolbar', this.api.table().container());
        tableToolbar.find('.dt-buttons').remove();
    }
}

(function () {
    function _init(settings) {
        const options = settings.oInit.toolbar;
        settings._toolbar = new TableToolbar(settings, options);
    }

    // Entry point for initializing the extension
    $(document).on('preInit.dt', function (e, settings) {
        if (e.namespace !== 'dt') {
            return;
        }

        if (settings.oInit.toolbar && !settings._toolbar) {
            _init(settings);
        }
    });
}());

// Toolbar renders (buttons)

DataTable.ext.buttons.toolbarCaption = {
    extend: 'spacer',
    text: 'Table caption',
    style: 'data-table-caption',
};

DataTable.ext.buttons.toolbarSpacer = {
    extend: 'spacer',
    style: 'data-table-spacer',
};

DataTable.ext.buttons.toolbarColumnVisibility = {
    extend: 'colvis',
    columns: 'th:not(.dt-select)',
    text: 'Вид',
    fade: 50,
    init: function(dt, node, config) {
        const icon = document.createElement('i');
        icon.className = 'rc-icon rc-icon--columns';
        node.prepend(icon);
        const iconArrow = document.createElement('i');
        iconArrow.className = 'rc-icon rc-icon--chevron-down';
        node.append(iconArrow);
    },
};

DataTable.ext.buttons.toolbarAction = {
    init: function(dt, node, config) {
        if (config.iconClass) {
            const icon = document.createElement('i');
            icon.className = 'rc-icon ' + config.iconClass;
            node.prepend(icon);
        }
    },
};

DataTable.ext.buttons.toolbarBatchAction = {
    init: function(dt, node, config) {
        if (config.iconClass) {
            const icon = document.createElement('i');
            icon.className = 'rc-icon ' + config.iconClass;
            node.prepend(icon);
        }
    },
};

DataTable.ext.buttons.toolbarMenu = {
    extend: 'collection',
    text: '',
    autoClose: true,
    background: false,
    fade: 50,
    init: function(dt, node, config) {
        const iconDots = document.createElement('i');
        iconDots.className = 'rc-icon rc-icon--dots';
        node.prepend(iconDots);
        const iconArrow = document.createElement('i');
        iconArrow.className = 'rc-icon rc-icon--chevron-down';
        node.append(iconArrow);
    },
};
