local http = require "http";
local shortport = require "shortport";
local stdnse = require "stdnse";
local vulns = require "vulns";

description = [[
Detects a denial of service vulnerability in the way the web server
handles requests for multiple overlapping/simple ranges of a page.
]]

--@usage
-- nmap -sV -p 8888 --script http-range-ddos localhost

--@output
--http-range-ddos: 
-- VULNERABLE:
-- Web server byterange filter DoS
--State: VULNERABLE
--The web server is vulnerable to a denial of service attack when numerous
--overlapping byte ranges are requested
--Disclosure date: 2022-05-30

author = "Petrov Aleksey"
license = "Same as Nmap--See https://nmap.org/book/man-legal.html" --* 
categories = {"vuln", "safe"}

local use_portrule = stdnse.get_script_args("stdnse.use_portrule") or false

if (use_portrule == "true") then
	portrule = shortport.http 
else portrule = function(host,port) return true end
end

action = function(host, port)
    local vuln = {
        title = 'Web server byterange filter DoS',
        state = vulns.STATE.NOT_VULN,
        --* IDS
        -- Changed description from specific server(Apache) to web server
        description = [[
        The web server is vulnerable to a denial of service attack when numerous
        overlapping byte ranges are requested.]],
        dates = {
            disclosure = {year = '2022', month = '05', day = '30'}
        };
    };
    local vuln_report = vulns.Report:new(SCRIPT_NAME, host, port);
    local hostname, path = stdnse.get_script_args('http-range-ddos.hostname', 'http-range-ddos.path');

    if not path then 
        path = '/';

        stdnse.debug1("Setting the request path to '/' since 'http-range-ddos.path' argument is missing.");
    end

    local request_opts = {
        header = {
            Range = "bytes=0-100",
            Connection = "close"
        },
        bypass_cache = true
    }

    if hostname then
        request_opts.header.Host = hostname;
    end

    local response = http.head(host, port, path, request_opts);
    
    if not response.status then
        stdnse.debug1("Functionality check HEAD request failed for %s (with path '%s').", hostname or host.ip, path);
    elseif response.status == 206 then
        -- Changed range relative to script http-vuln-cve2011-3192
        request_opts.header.Range = "bytes=0-0,1-1,2-2,3-3,4-4,5-5,6-6,7-7,8-8,9-9,10-10,11-11,12-12,13-13,14-14,15-15,16-16,17-17,18-18,15-15,15-15,21-21,22-22,22-100000";

        response = http.head(host, port, path, request_opts);
       
        if not response.status then
            stdnse.debug1("Invalid response from server to the vulnerability check");
        elseif response.status == 206 then
            vuln.state = vulns.STATE.VULN;
        else
            stdnse.debug1("Server isn't vulnerable (%i status code)", response.status);
        end
    else
        stdnse.debug1("Server ignores the range header (%i status code)", response.status);
    end

    return vuln_report: make_output(vuln);
end